/**
 * UCS Player Initialization Script
 */
document.addEventListener( 'DOMContentLoaded', function () {
    if ( typeof ucsPlayerQueue !== 'undefined' && Array.isArray( ucsPlayerQueue ) ) {
        ucsPlayerQueue.forEach( function ( playerConfig ) {
            initUcsPlayer( playerConfig.id, playerConfig.config );
        } );
    }
} );

function initUcsPlayer( playerId, config ) {
    var video = document.getElementById( playerId );
    var wrapper = document.getElementById( 'wrapper-' + playerId );
    var topSlot = document.getElementById( 'ucs-top-slot-' + playerId );
    var bottomSlot = document.getElementById( 'ucs-bottom-slot-' + playerId );
    
    var adOverlay = document.getElementById( 'ad-overlay-' + playerId );
    var adImg = document.getElementById( 'ad-img-' + playerId );
    var adSkipBtn = document.getElementById( 'ad-skip-' + playerId );
    var gateOverlay = document.getElementById( 'gate-' + playerId );
    
    if ( ! video ) return;

    var streamUrl = config.streamUrl;
    var ads = config.ads || [];
    var topAds = config.topAds || [];
    var bottomAds = config.bottomAds || [];
    var adConfig = config.adConfig || {};
    var timeLimit = config.timeLimit || 0;
    
    var isAdPlaying = false;
    var lastMidrollTime = 0;
    var viewCount = parseInt( localStorage.getItem('ucs_view_count') || '0' ) + 1;
    localStorage.setItem('ucs_view_count', viewCount);

    // --- ROTATORS ---
    function startSlotRotator(slotEl, adsList) {
        if (!slotEl || adsList.length === 0) return;
        slotEl.style.display = 'block';
        var idx = 0;
        var showAd = function() {
            var ad = adsList[idx];
            if (ad && ad.url) { // Safety check
                var html = '';
                if (ad.type === 'image') {
                    html = '<a href="'+ad.link+'" target="_blank"><img src="'+ad.url+'" style="max-width:100%; height:auto;"></a>';
                } else if (ad.type === 'video') {
                    html = '<a href="'+ad.link+'" target="_blank"><video src="'+ad.url+'" autoplay muted loop style="max-width:100%;"></video></a>';
                }
                slotEl.innerHTML = html;
            }
            idx = (idx + 1) % adsList.length;
        };
        showAd();
        if (adConfig.rotationFreq > 0 && adsList.length > 1) {
            setInterval(showAd, adConfig.rotationFreq * 1000);
        }
    }
    startSlotRotator(topSlot, topAds);
    startSlotRotator(bottomSlot, bottomAds);

    // --- AD LOGIC ---
    function shouldShowAd() {
        if ( ads.length === 0 ) return false;
        var freq = adConfig.viewFrequency || 1;
        return ( viewCount % freq === 0 );
    }

    function playAd( type ) {
        if ( ! shouldShowAd() ) {
            if ( type === 'preroll' ) loadMainStream();
            return;
        }

        var ad = ads[Math.floor(Math.random() * ads.length)];
        
        // SAFETY CHECK: If ad is invalid, skip immediately
        if (!ad || !ad.url || (ad.type !== 'video' && ad.type !== 'image')) {
            console.warn('UCS: Invalid ad skipped');
            if ( type === 'preroll' ) loadMainStream();
            return;
        }

        isAdPlaying = true;
        adOverlay.style.display = 'flex';
        adSkipBtn.style.display = 'block';
        adSkipBtn.disabled = true;
        var skipTime = parseInt(ad.skip || 5);
        var timeLeft = skipTime;

        adSkipBtn.innerText = 'Skip in ' + timeLeft + '...';
        var timerInt = setInterval(function() {
            timeLeft--;
            if ( timeLeft <= 0 ) {
                clearInterval(timerInt);
                adSkipBtn.innerText = 'Skip Ad';
                adSkipBtn.disabled = false;
            } else {
                adSkipBtn.innerText = 'Skip in ' + timeLeft + '...';
            }
        }, 1000);

        adSkipBtn.onclick = function() { clearInterval(timerInt); endAd(); };

        if ( ad.type === 'video' ) {
            video.src = ad.url;
            video.play().catch(e => console.error('Ad play failed', e));
            video.onended = endAd;
            wrapper.style.cursor = 'pointer';
            wrapper.onclick = function(e) { if(e.target !== adSkipBtn) window.open(ad.link, '_blank'); };
        } else if ( ad.type === 'image' ) {
            video.pause();
            adImg.src = ad.url;
            adImg.style.display = 'block';
            setTimeout(endAd, 10000); 
            adImg.style.cursor = 'pointer';
            adImg.onclick = function() { window.open(ad.link, '_blank'); };
        }
    }

    function endAd() {
        isAdPlaying = false;
        adOverlay.style.display = 'none';
        adImg.style.display = 'none';
        wrapper.onclick = null;
        if ( video.src !== streamUrl ) { loadMainStream(); } else { video.play(); }
    }

    function loadMainStream() {
        if ( Hls.isSupported() ) {
            var hls = new Hls();
            hls.loadSource( streamUrl );
            hls.attachMedia( video );
            hls.on(Hls.Events.MANIFEST_PARSED, function() {
                if ( config.autoplay ) video.play().catch(()=>{});
            });
        } else if ( video.canPlayType( 'application/vnd.apple.mpegurl' ) ) {
            video.src = streamUrl;
            if ( config.autoplay ) video.play().catch(()=>{});
        }
    }

    // Init
    playAd('preroll');

    // Timed Events
    video.addEventListener('timeupdate', function() {
        if ( isAdPlaying ) return;
        if ( adConfig.midrollInterval > 0 ) {
            if ( video.currentTime > 0 && video.currentTime % adConfig.midrollInterval < 1 && video.currentTime > lastMidrollTime + 10 ) {
                lastMidrollTime = video.currentTime;
                playAd('midroll');
            }
        }
        if ( timeLimit > 0 && video.currentTime >= timeLimit ) {
            video.pause();
            video.currentTime = 0;
            if ( gateOverlay ) {
                gateOverlay.style.display = 'flex';
                video.controls = false;
            }
        }
    });
}