<?php
/**
 * Handles the Admin Menu, Settings Pages, and Importers.
 *
 * @package UCS_Streamer
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class UCS_Admin {

	/**
	 * Initialize hooks
	 */
	public static function init() {
		// Priority 20 ensures we run after the CPT is registered
		add_action( 'admin_menu', array( __CLASS__, 'add_plugin_admin_menu' ), 20 );
		add_action( 'admin_init', array( __CLASS__, 'register_settings' ) );
		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_admin_scripts' ) );
	}

	/**
	 * Adds Settings to the main 'UCS Streamer' menu.
	 */
	public static function add_plugin_admin_menu() {
		// This MUST match the slug of your Custom Post Type menu
		$parent_slug = 'edit.php?post_type=channel';

		// 1. Settings
		add_submenu_page(
			$parent_slug,
			'UCS Settings',    // Page Title
			'Settings',        // Menu Title
			'manage_options',  // Capability
			'ucs-settings',    // Menu Slug
			array( __CLASS__, 'render_settings_page' ) // Callback
		);

		// 2. Importers
		add_submenu_page(
			$parent_slug,
			'Import Channels',
			'Importers',
			'manage_options',
			'ucs-importers',
			array( 'UCS_Importer', 'render_importer_page' )
		);

		// 3. Analytics
		add_submenu_page(
			$parent_slug,
			'Analytics',
			'Analytics',
			'manage_options',
			'ucs-analytics',
			array( __CLASS__, 'render_analytics_page' )
		);
		
		// 4. Instructions
		add_submenu_page(
			$parent_slug,
			'How to Use',
			'Instructions',
			'manage_options',
			'ucs-instructions',
			array( __CLASS__, 'render_instructions_page' )
		);
	}

	/**
	 * Enqueue admin scripts.
	 */
	public static function enqueue_admin_scripts( $hook ) {
		// Robust check to see if we are on a plugin page
		if ( strpos( $hook, 'ucs-' ) === false && strpos( $hook, 'channel' ) === false ) {
			return;
		}

		wp_enqueue_style( 'ucs-admin-css', UCS_PLUGIN_URL . 'assets/css/admin-settings.css', array(), UCS_VERSION );
		wp_enqueue_script( 'ucs-admin-js', UCS_PLUGIN_URL . 'assets/js/admin-settings.js', array( 'jquery' ), UCS_VERSION, true );
	}

	/**
	 * Register settings.
	 */
	public static function register_settings() {
		register_setting( 'ucs_settings_group', 'ucs_options', array( __CLASS__, 'sanitize_settings' ) );

		// 1. General Section
		add_settings_section( 'ucs_section_general', 'Access Control', array( __CLASS__, 'section_general_cb' ), 'ucs-settings' );
		add_settings_field( 'access_mode', 'Global Access Mode', array( __CLASS__, 'render_field_access_mode' ), 'ucs-settings', 'ucs_section_general', ['id' => 'access_mode'] );
		add_settings_field( 'access_product_id', 'WooCommerce Product ID', array( __CLASS__, 'render_field_text' ), 'ucs-settings', 'ucs_section_general', ['id' => 'access_product_id', 'class' => 'small-text'] );
		add_settings_field( 'access_preview_time', 'Preview Time (Seconds)', array( __CLASS__, 'render_field_text' ), 'ucs-settings', 'ucs_section_general', ['id' => 'access_preview_time', 'class' => 'small-text'] );
		
		// 2. Player Section
		add_settings_section( 'ucs_section_player', 'Player Settings', array( __CLASS__, 'section_player_cb' ), 'ucs-settings' );
		add_settings_field( 'player_autoplay', 'Autoplay', array( __CLASS__, 'render_field_checkbox' ), 'ucs-settings', 'ucs_section_player', ['id' => 'player_autoplay', 'label' => 'Start playing video on load'] );

		// 3. License Section
		add_settings_section( 'ucs_section_license', 'License Activation', array( __CLASS__, 'render_license_section_text' ), 'ucs-settings' );
		add_settings_field( 'license_key', 'License Key', array( __CLASS__, 'render_field_text' ), 'ucs-settings', 'ucs_section_license', ['id' => 'license_key', 'class' => 'regular-text'] );

		// 4. Monetization Section
		add_settings_section( 'ucs_section_monetization', 'Monetization (Premium)', array( __CLASS__, 'render_monetization_section_text' ), 'ucs-settings' );
		add_settings_field( 'ad_midroll_interval', 'Mid-Roll Interval (Minutes)', array( __CLASS__, 'render_field_text' ), 'ucs-settings', 'ucs_section_monetization', ['id' => 'ad_midroll_interval', 'class' => 'small-text', 'desc' => 'Show an ad every X minutes. Set to 0 to disable mid-rolls.'] );
        add_settings_field( 'ad_view_frequency', 'Ad Frequency (Channel Views)', array( __CLASS__, 'render_field_text' ), 'ucs-settings', 'ucs_section_monetization', ['id' => 'ad_view_frequency', 'class' => 'small-text', 'desc' => 'Show ads only after every X channel changes/views. Set to 1 for every view.'] );
        add_settings_field( 'ad_enable_rotation', 'Enable Rotation', array( __CLASS__, 'render_field_checkbox' ), 'ucs-settings', 'ucs_section_monetization', ['id' => 'ad_enable_rotation', 'label' => 'Randomly rotate all active ads matching the channel category/country.'] );
	}

    // --- Section Callbacks (Text for top of sections) ---
    public static function section_general_cb() { echo '<p>Configure who can access your streams.</p>'; }
    public static function section_player_cb() { echo '<p>Customize the frontend video player.</p>'; }

	// --- Field Renderers ---
	public static function render_field_text( $args ) {
		$options = get_option( 'ucs_options', array() );
		$value = isset( $options[ $args['id'] ] ) ? $options[ $args['id'] ] : '';
		$disabled = isset( $args['disabled'] ) && $args['disabled'] ? 'disabled' : '';
		echo '<input type="text" id="' . esc_attr( $args['id'] ) . '" name="ucs_options[' . esc_attr( $args['id'] ) . ']" value="' . esc_attr( $value ) . '" class="' . esc_attr( $args['class'] ) . '" ' . $disabled . ' />';
		if ( ! empty( $args['desc'] ) ) echo ' <p class="description">' . esc_html( $args['desc'] ) . '</p>';
	}

	public static function render_field_checkbox( $args ) {
		$options = get_option( 'ucs_options', array() );
		$checked = isset( $options[ $args['id'] ] ) ? $options[ $args['id'] ] : 0;
		$disabled = isset( $args['disabled'] ) && $args['disabled'] ? 'disabled' : '';
		echo '<label><input type="checkbox" id="' . esc_attr( $args['id'] ) . '" name="ucs_options[' . esc_attr( $args['id'] ) . ']" value="1" ' . checked( 1, $checked, false ) . ' ' . $disabled . ' /> ' . esc_html( $args['label'] ) . '</label>';
	}
	
	public static function render_field_access_mode( $args ) {
		$options = get_option( 'ucs_options', array() );
		$current = isset( $options['access_mode'] ) ? $options['access_mode'] : 'open';
		?>
		<select name="ucs_options[access_mode]" id="access_mode">
			<option value="open" <?php selected( $current, 'open' ); ?>>Open Access (Free for All)</option>
			<option value="login" <?php selected( $current, 'login' ); ?>>Logged-In Required</option>
			<option value="timed" <?php selected( $current, 'timed' ); ?>>Timed Soft Gating (Free Trial)</option>
			<option value="paid" <?php selected( $current, 'paid' ); ?>>Paid Hard Gating (WooCommerce)</option>
		</select>
		<p class="description">Select how users access content.</p>
		<?php
	}

	public static function render_license_section_text() {
		echo '<p>Enter your license key to unlock premium features.</p>';
		$status = get_option('ucs_license_status');
		if ( $status === 'active' ) {
			echo '<p style="color: green;"><strong>License is active.</strong></p>';
		} else {
			echo '<p style="color: orange;"><strong>License is inactive.</strong></p>';
		}
	}

	public static function render_monetization_section_text() {
		if ( ! class_exists('UCS_License') || ! UCS_License::is_license_active() ) {
			echo '<p style="font-weight: bold; color: #c00;">This is a premium feature. Please activate your license.</p>';
		} else {
			echo '<p>Ads created in "Ad Creatives" will be automatically rotated based on these rules and their specific targeting options.</p>';
		}
	}
	
	public static function sanitize_settings( $input ) {
		$sanitized_input = array();
		$old_options = get_option('ucs_options');

		// Access Control
		$sanitized_input['access_mode'] = sanitize_text_field( $input['access_mode'] ?? 'open' );
		$sanitized_input['access_product_id'] = absint( $input['access_product_id'] ?? 0 );
		$sanitized_input['access_preview_time'] = absint( $input['access_preview_time'] ?? 60 );

		// License
		$new_key = isset($input['license_key']) ? sanitize_text_field($input['license_key']) : '';
		$old_key = isset($old_options['license_key']) ? $old_options['license_key'] : '';

		if ( class_exists('UCS_License') ) {
			if ( ! empty( $new_key ) && $new_key !== $old_key ) {
				UCS_License::activate_license( $new_key );
			} elseif ( empty( $new_key ) && ! empty( $old_key ) ) {
				UCS_License::deactivate_license();
			}
		}
		$sanitized_input['license_key'] = $new_key;
		
		$sanitized_input['player_autoplay'] = ( isset( $input['player_autoplay'] ) && $input['player_autoplay'] == '1' ) ? 1 : 0;

		// Monetization
		if ( class_exists('UCS_License') && UCS_License::is_license_active() ) {
            $sanitized_input['ad_midroll_interval'] = absint($input['ad_midroll_interval'] ?? 0);
            $sanitized_input['ad_view_frequency'] = absint($input['ad_view_frequency'] ?? 1);
            $sanitized_input['ad_enable_rotation'] = isset($input['ad_enable_rotation']) ? 1 : 0;
		} else {
			// Preserve old values
            $sanitized_input['ad_midroll_interval'] = isset($old_options['ad_midroll_interval']) ? $old_options['ad_midroll_interval'] : 0;
            $sanitized_input['ad_view_frequency'] = isset($old_options['ad_view_frequency']) ? $old_options['ad_view_frequency'] : 1;
            $sanitized_input['ad_enable_rotation'] = isset($old_options['ad_enable_rotation']) ? $old_options['ad_enable_rotation'] : 0;
		}

		return $sanitized_input;
	}

    // --- THE FIX: Properly separated rendering ---
	public static function render_settings_page() {
		?>
		<div class="wrap ucs-settings-wrap">
			<h1>UCS Streamer Settings</h1>
			
            <!-- The Tabs -->
			<h2 class="nav-tab-wrapper ucs-nav-tabs">
				<a href="#ucs_section_general" class="nav-tab nav-tab-active">General</a>
				<a href="#ucs_section_player" class="nav-tab">Player</a>
				<a href="#ucs_section_license" class="nav-tab">License</a>
				<a href="#ucs_section_monetization" class="nav-tab">Monetization</a>
			</h2>
            
			<form method="post" action="options.php" class="ucs-settings-form">
				<?php settings_fields( 'ucs_settings_group' ); ?>
				
                <!-- General Tab Content -->
				<div id="ucs_section_general" class="ucs-tab-content active">
					<h3>Access Control & General Settings</h3>
					<table class="form-table">
                        <?php do_settings_fields( 'ucs-settings', 'ucs_section_general' ); ?>
                    </table>
				</div>

                <!-- Player Tab Content -->
				<div id="ucs_section_player" class="ucs-tab-content">
					<h3>Player Settings</h3>
					<table class="form-table">
                        <?php do_settings_fields( 'ucs-settings', 'ucs_section_player' ); ?>
                    </table>
				</div>
                
                <!-- License Tab Content -->
				<div id="ucs_section_license" class="ucs-tab-content">
					<h3>License Activation</h3>
                    <?php self::render_license_section_text(); ?>
					<table class="form-table">
                        <?php do_settings_fields( 'ucs-settings', 'ucs_section_license' ); ?>
                    </table>
				</div>

                <!-- Monetization Tab Content -->
				<div id="ucs_section_monetization" class="ucs-tab-content">
					<h3>Monetization</h3>
					<?php 
                    self::render_monetization_section_text();
					if ( class_exists('UCS_License') && UCS_License::is_license_active() ) {
						echo '<table class="form-table">';
						do_settings_fields( 'ucs-settings', 'ucs_section_monetization' );
						echo '</table>';
					}
					?>
				</div>
				<?php submit_button(); ?>
			</form>
		</div>
		<?php
	}

	public static function render_analytics_page() {
        // Get counts
        $channel_count = wp_count_posts('channel')->publish;
        $ad_count = wp_count_posts('ucs_ad_creative')->publish;
        $cat_count = wp_count_terms(['taxonomy' => 'channel_category']);
        $country_count = wp_count_terms(['taxonomy' => 'channel_country']);
		?>
        <div class="wrap">
            <h1><?php _e('Channel Analytics', 'ucs-streamer'); ?></h1>
            <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 20px; margin-top: 20px;">
                <div style="background: #fff; padding: 20px; border: 1px solid #ccc; text-align: center;">
                    <h3 style="margin: 0;">Total Channels</h3>
                    <p style="font-size: 2rem; margin: 10px 0; font-weight: bold;"><?php echo intval($channel_count); ?></p>
                </div>
                <div style="background: #fff; padding: 20px; border: 1px solid #ccc; text-align: center;">
                    <h3 style="margin: 0;">Active Ads</h3>
                    <p style="font-size: 2rem; margin: 10px 0; font-weight: bold;"><?php echo intval($ad_count); ?></p>
                </div>
                <div style="background: #fff; padding: 20px; border: 1px solid #ccc; text-align: center;">
                    <h3 style="margin: 0;">Categories</h3>
                    <p style="font-size: 2rem; margin: 10px 0; font-weight: bold;"><?php echo intval($cat_count); ?></p>
                </div>
                <div style="background: #fff; padding: 20px; border: 1px solid #ccc; text-align: center;">
                    <h3 style="margin: 0;">Countries</h3>
                    <p style="font-size: 2rem; margin: 10px 0; font-weight: bold;"><?php echo intval($country_count); ?></p>
                </div>
            </div>
            <p style="margin-top: 30px;"><em>(Advanced viewer tracking and ad impression data will be available in a future update)</em></p>
        </div>
        <?php
	}

	public static function render_instructions_page() {
        ?>
		<div class="wrap">
            <h1>How to Use UCS Streamer</h1>
            
            <h2>1. Managing Channels</h2>
            <p>Go to <strong>UCS Streamer > All Channels</strong> to view your list.</p>
            <p>Use <strong>Import Channels</strong> to upload CSV/M3U lists.</p>

            <h2>2. Displaying Channels (Frontend)</h2>
            <p><strong>Single Channel:</strong> Use the shortcode <code>[ucs_player id="123"]</code>.</p>
            <p><strong>Channel Directory (New):</strong> Use the shortcode <code>[ucs_directory]</code> to display a searchable, filterable grid of all your channels on any page.</p>

            <h2>3. Ads (Premium)</h2>
            <p>Go to <strong>UCS Streamer > Ad Creatives</strong> to upload MP4 video ads or image banners.</p>
            <p>Then go to <strong>UCS Streamer > Settings > Monetization</strong> to select your default pre-roll ad and configure ad slots above/below the player.</p>
        </div>
        <?php
	}
}