<?php
/**
 * Registers the 'ucs_ad_creative' Custom Post Type and meta boxes.
 *
 * @package UCS_Streamer
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class UCS_Ads {

	/**
	 * Initialize hooks
	 */
	public static function init() {
		add_action( 'init', array( __CLASS__, 'register_post_type' ) );
		add_action( 'add_meta_boxes', array( __CLASS__, 'add_meta_boxes' ) );
		add_action( 'save_post_ucs_ad_creative', array( __CLASS__, 'save_meta_boxes' ) );
	}

	/**
	 * Register the 'ucs_ad_creative' CPT.
	 */
	public static function register_post_type() {
		$labels = array(
			'name'                  => _x( 'Ad Creatives', 'Post Type General Name', 'ucs-streamer' ),
			'singular_name'         => _x( 'Ad Creative', 'Post Type Singular Name', 'ucs-streamer' ),
			'menu_name'             => __( 'Ad Creatives', 'ucs-streamer' ),
			'name_admin_bar'        => __( 'Ad Creative', 'ucs-streamer' ),
			'add_new'               => __( 'Add New Ad', 'ucs-streamer' ),
			'add_new_item'          => __( 'Add New Ad Creative', 'ucs-streamer' ),
			'new_item'              => __( 'New Ad Creative', 'ucs-streamer' ),
			'edit_item'             => __( 'Edit Ad Creative', 'ucs-streamer' ),
			'view_item'             => __( 'View Ad Creative', 'ucs-streamer' ),
			'all_items'             => __( 'All Ad Creatives', 'ucs-streamer' ),
			'search_items'          => __( 'Search Ads', 'ucs-streamer' ),
			'not_found'             => __( 'No ads found.', 'ucs-streamer' ),
			'not_found_in_trash'    => __( 'No ads found in Trash.', 'ucs-streamer' ),
		);
		$args = array(
			'label'                 => __( 'Ad Creative', 'ucs-streamer' ),
			'description'           => __( 'Manage video and image ads for monetization.', 'ucs-streamer' ),
			'labels'                => $labels,
			'supports'              => array( 'title' ),
			'hierarchical'          => false,
			'public'                => false,
			'show_ui'               => true,
			'show_in_menu'          => 'edit.php?post_type=channel',
			'menu_icon'             => 'dashicons-megaphone',
			'show_in_admin_bar'     => false,
			'show_in_nav_menus'     => false,
			'can_export'            => true,
			'has_archive'           => false,
			'exclude_from_search'   => true,
			'publicly_queryable'    => false,
			'capability_type'       => 'post',
			'show_in_rest'          => true,
		);
		register_post_type( 'ucs_ad_creative', $args );
	}

	/**
	 * Add meta boxes
	 */
	public static function add_meta_boxes() {
		add_meta_box( 
			'ucs_ad_details', 
			__( 'Ad Configuration', 'ucs-streamer' ), 
			array( __CLASS__, 'render_meta_box' ), 
			'ucs_ad_creative', 
			'normal', 
			'high' 
		);
		
		add_meta_box( 
			'ucs_ad_targeting', 
			__( 'Ad Targeting (Optional)', 'ucs-streamer' ), 
			array( __CLASS__, 'render_targeting_meta_box' ), 
			'ucs_ad_creative', 
			'normal', 
			'high' 
		);
	}

	/**
	 * Render Configuration meta box
	 */
	public static function render_meta_box( $post ) {
		wp_nonce_field( 'ucs_save_ad_details', 'ucs_ad_details_nonce' );

		$ad_type = get_post_meta( $post->ID, '_ucs_ad_type', true );
		$ad_url = get_post_meta( $post->ID, '_ucs_ad_url', true );
		$ad_link = get_post_meta( $post->ID, '_ucs_ad_link', true );
		$ad_skip = get_post_meta( $post->ID, '_ucs_ad_skip', true );

		if ( empty( $ad_skip ) && $ad_skip !== '0' ) $ad_skip = 5; 
		?>
		<table class="form-table">
			<tr valign="top">
				<th scope="row"><label for="ucs_ad_type"><?php _e( 'Ad Type', 'ucs-streamer' ); ?></label></th>
				<td>
					<select name="ucs_ad_type" id="ucs_ad_type">
						<option value="video" <?php selected( $ad_type, 'video' ); ?>>Video File (.mp4)</option>
						<option value="image" <?php selected( $ad_type, 'image' ); ?>>Image Banner (.jpg, .png)</option>
					</select>
				</td>
			</tr>
			<tr valign="top">
				<th scope="row"><label for="ucs_ad_url"><?php _e( 'Media URL', 'ucs-streamer' ); ?></label></th>
				<td>
					<input type="url" id="ucs_ad_url" name="ucs_ad_url" value="<?php echo esc_url( $ad_url ); ?>" class="widefat" placeholder="http://..." />
					<p class="description"><?php _e('Direct link to the MP4 video or Image file.', 'ucs-streamer'); ?></p>
				</td>
			</tr>
			<tr valign="top">
				<th scope="row"><label for="ucs_ad_link"><?php _e( 'Destination URL (Click-through)', 'ucs-streamer' ); ?></label></th>
				<td>
					<input type="url" id="ucs_ad_link" name="ucs_ad_link" value="<?php echo esc_url( $ad_link ); ?>" class="widefat" placeholder="http://..." />
					<p class="description"><?php _e('Where the user goes if they click the ad.', 'ucs-streamer'); ?></p>
				</td>
			</tr>
			<tr valign="top">
				<th scope="row"><label for="ucs_ad_skip"><?php _e( 'Skip Button appears after (seconds)', 'ucs-streamer' ); ?></label></th>
				<td>
					<input type="number" id="ucs_ad_skip" name="ucs_ad_skip" value="<?php echo esc_attr( $ad_skip ); ?>" class="small-text" min="0" />
					<p class="description"><?php _e('Set to 0 to make unskippable (use with caution). Default is 5.', 'ucs-streamer'); ?></p>
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Render Targeting meta box
	 */
	public static function render_targeting_meta_box( $post ) {
		$target_cats = get_post_meta( $post->ID, '_ucs_target_cats', true );
		$target_countries = get_post_meta( $post->ID, '_ucs_target_countries', true );

		// Get all categories
		$categories = get_terms( array( 'taxonomy' => 'channel_category', 'hide_empty' => false ) );
		// Get all countries
		$countries = get_terms( array( 'taxonomy' => 'channel_country', 'hide_empty' => false ) );
		?>
		<p><em><?php _e('Leave blank to show on ALL channels.', 'ucs-streamer'); ?></em></p>
		<table class="form-table">
			<tr valign="top">
				<th scope="row"><?php _e('Target Categories', 'ucs-streamer'); ?></th>
				<td>
					<?php if ( ! empty( $categories ) && ! is_wp_error( $categories ) ) : ?>
						<div style="max-height: 150px; overflow-y: auto; border: 1px solid #ddd; padding: 10px;">
							<?php foreach ( $categories as $cat ) : ?>
								<label style="display:block;">
									<input type="checkbox" name="ucs_target_cats[]" value="<?php echo esc_attr( $cat->slug ); ?>" <?php if ( is_array( $target_cats ) && in_array( $cat->slug, $target_cats ) ) echo 'checked'; ?> />
									<?php echo esc_html( $cat->name ); ?>
								</label>
							<?php endforeach; ?>
						</div>
					<?php else : ?>
						<p><?php _e('No categories found.', 'ucs-streamer'); ?></p>
					<?php endif; ?>
				</td>
			</tr>
			<tr valign="top">
				<th scope="row"><?php _e('Target Countries', 'ucs-streamer'); ?></th>
				<td>
					<?php if ( ! empty( $countries ) && ! is_wp_error( $countries ) ) : ?>
						<div style="max-height: 150px; overflow-y: auto; border: 1px solid #ddd; padding: 10px;">
							<?php foreach ( $countries as $country ) : ?>
								<label style="display:block;">
									<input type="checkbox" name="ucs_target_countries[]" value="<?php echo esc_attr( $country->slug ); ?>" <?php if ( is_array( $target_countries ) && in_array( $country->slug, $target_countries ) ) echo 'checked'; ?> />
									<?php echo esc_html( $country->name ); ?>
								</label>
							<?php endforeach; ?>
						</div>
					<?php else : ?>
						<p><?php _e('No countries found.', 'ucs-streamer'); ?></p>
					<?php endif; ?>
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Save meta box
	 */
	public static function save_meta_boxes( $post_id ) {
		if ( ! isset( $_POST['ucs_ad_details_nonce'] ) || ! wp_verify_nonce( $_POST['ucs_ad_details_nonce'], 'ucs_save_ad_details' ) ) return;
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) return;
		if ( ! current_user_can( 'edit_post', $post_id ) ) return;

		// Config
		if ( isset( $_POST['ucs_ad_type'] ) ) update_post_meta( $post_id, '_ucs_ad_type', sanitize_text_field( $_POST['ucs_ad_type'] ) );
		if ( isset( $_POST['ucs_ad_url'] ) ) update_post_meta( $post_id, '_ucs_ad_url', esc_url_raw( $_POST['ucs_ad_url'] ) );
		if ( isset( $_POST['ucs_ad_link'] ) ) update_post_meta( $post_id, '_ucs_ad_link', esc_url_raw( $_POST['ucs_ad_link'] ) );
		if ( isset( $_POST['ucs_ad_skip'] ) ) update_post_meta( $post_id, '_ucs_ad_skip', absint( $_POST['ucs_ad_skip'] ) );

		// Targeting
		$cats = isset( $_POST['ucs_target_cats'] ) ? array_map( 'sanitize_text_field', $_POST['ucs_target_cats'] ) : array();
		update_post_meta( $post_id, '_ucs_target_cats', $cats );

		$countries = isset( $_POST['ucs_target_countries'] ) ? array_map( 'sanitize_text_field', $_POST['ucs_target_countries'] ) : array();
		update_post_meta( $post_id, '_ucs_target_countries', $countries );
	}
}