<?php
/**
 * Registers the Gutenberg Block
 *
 * @package UCS_Streamer
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class UCS_Block {

	/**
	 * Initialize hooks
	 */
	public static function init() {
		add_action( 'init', array( __CLASS__, 'register_block' ) );
		add_action( 'rest_api_init', array( __CLASS__, 'register_search_route' ) );
	}

	/**
	 * Register the dynamic block.
	 */
	public static function register_block() {
		// Register the block type from block.json
		register_block_type( UCS_PLUGIN_DIR . 'build/' );
	}

	/**
	 * Register the REST API route for searching channels.
	 * /wp-json/ucs/v1/search?q=...
	 */
	public static function register_search_route() {
		register_rest_route( 'ucs/v1', '/search', array(
			'methods'             => WP_REST_Server::READABLE,
			'callback'            => array( __CLASS__, 'rest_search_channels' ),
			'permission_callback' => function() {
				// Only users who can edit posts can search
				return current_user_can( 'edit_posts' );
			},
			'args' => array(
				'q' => array(
					'validate_callback' => function( $param ) {
						return is_string( $param );
					},
					'sanitize_callback' => 'sanitize_text_field',
				),
			),
		) );
	}

	/**
	 * Callback for the REST search.
	 *
	 * @param WP_REST_Request $request
	 * @return WP_REST_Response
	 */
	public static function rest_search_channels( $request ) {
		$query = $request['q'];
		$results = array();

		if ( empty( $query ) ) {
			return new WP_REST_Response( $results, 200 );
		}

		$args = array(
			'post_type'      => 'channel',
			'post_status'    => 'publish',
			'posts_per_page' => 20,
			's'              => $query, // Search by title
		);

		$search_query = new WP_Query( $args );

		if ( $search_query->have_posts() ) {
			while ( $search_query->have_posts() ) {
				$search_query->the_post();
				$results[] = array(
					'id'    => get_the_ID(),
					'title' => get_the_title(),
				);
			}
		}
		wp_reset_postdata();

		return new WP_REST_Response( $results, 200 );
	}
	
	/**
	 * Render callback for the dynamic block.
	 *
	 * @param array $attributes Block attributes.
	 * @return string HTML output.
	 */
	public static function render_block( $attributes ) {
		$channel_id = $attributes['channelId'] ?? 0;

		if ( empty( $channel_id ) ) {
			return '<div class="ucs-block-placeholder"><p>' . __( 'Please select a channel in the editor.', 'ucs-streamer' ) . '</p></div>';
		}

		// Reuse the shortcode rendering logic
		if ( class_exists( 'UCS_Frontend' ) ) {
			return UCS_Frontend::get_player_html( $channel_id );
		}
		return ''; // Return empty if class not found
	}
}