<?php
/**
 * Handles all frontend output: template loading, shortcodes, and scripts.
 *
 * @package UCS_Streamer
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class UCS_Frontend {

	public static function init() {
		add_action( 'wp_enqueue_scripts', array( __CLASS__, 'enqueue_scripts' ) );
		add_filter( 'template_include', array( __CLASS__, 'load_channel_template' ) );
		add_shortcode( 'ucs_player', array( __CLASS__, 'render_player_shortcode' ) );
		add_shortcode( 'ucs_directory', array( __CLASS__, 'render_directory_shortcode' ) );
	}

	public static function enqueue_scripts() {
		wp_register_script( 'ucs-hls-js', 'https://cdn.jsdelivr.net/npm/hls.js@1.5.11/dist/hls.min.js', array(), '1.5.11', true );
		wp_register_script( 'ucs-player-js', UCS_PLUGIN_URL . 'assets/js/player.js', array( 'ucs-hls-js' ), UCS_VERSION, true );
		wp_register_style( 'ucs-player-css', UCS_PLUGIN_URL . 'assets/css/player.css', array(), UCS_VERSION );
	}

	public static function load_channel_template( $template ) {
		if ( is_singular( 'channel' ) ) {
			$new_template = UCS_PLUGIN_DIR . 'templates/single-channel.php';
			if ( file_exists( $new_template ) ) return $new_template;
		}
		return $template;
	}

	public static function render_player_shortcode( $atts ) {
		$atts = shortcode_atts( array( 'id' => null, 'slug' => null ), $atts, 'ucs_player' );
		$post_id = $atts['id'];
		if ( ! $post_id && $atts['slug'] ) {
			$post = get_page_by_path( $atts['slug'], OBJECT, 'channel' );
			if ( $post ) $post_id = $post->ID;
		}
		if ( ! $post_id || get_post_type( $post_id ) !== 'channel' ) return '<p class="ucs-error">' . __( 'Error: Channel not found.', 'ucs-streamer' ) . '</p>';
		return self::get_player_html( $post_id );
	}

    public static function render_directory_shortcode( $atts ) {
        $search = isset($_GET['ucs_search']) ? sanitize_text_field($_GET['ucs_search']) : '';
        $cat = isset($_GET['ucs_cat']) ? sanitize_text_field($_GET['ucs_cat']) : '';
        $country = isset($_GET['ucs_country']) ? sanitize_text_field($_GET['ucs_country']) : '';

        $args = array(
            'post_type' => 'channel',
            'post_status' => 'publish',
            'posts_per_page' => 24,
            'paged' => get_query_var('paged') ? get_query_var('paged') : 1,
        );

        if ($search) $args['s'] = $search;
        $tax_query = [];
        if ($cat) $tax_query[] = ['taxonomy' => 'channel_category', 'field' => 'slug', 'terms' => $cat];
        if ($country) $tax_query[] = ['taxonomy' => 'channel_country', 'field' => 'slug', 'terms' => $country];
        if (!empty($tax_query)) $args['tax_query'] = $tax_query;

        $query = new WP_Query($args);

        ob_start();
        wp_enqueue_style('ucs-player-css');
        ?>
        <div class="ucs-directory">
            <form method="GET" class="ucs-directory-controls">
                <input type="text" name="ucs_search" placeholder="<?php esc_attr_e('Search channels...', 'ucs-streamer'); ?>" value="<?php echo esc_attr($search); ?>">
                <select name="ucs_cat"><option value=""><?php esc_html_e('All Categories', 'ucs-streamer'); ?></option>
					<?php $cats = get_terms(['taxonomy' => 'channel_category', 'hide_empty' => true]);
					if(!is_wp_error($cats)) foreach($cats as $c) echo '<option value="'.$c->slug.'" '.selected($cat, $c->slug).'>'.$c->name.'</option>'; ?>
                </select>
                <select name="ucs_country"><option value=""><?php esc_html_e('All Countries', 'ucs-streamer'); ?></option>
					<?php $countries = get_terms(['taxonomy' => 'channel_country', 'hide_empty' => true]);
					if(!is_wp_error($countries)) foreach($countries as $c) echo '<option value="'.$c->slug.'" '.selected($country, $c->slug).'>'.$c->name.'</option>'; ?>
                </select>
                <button type="submit" class="button button-primary"><?php esc_html_e('Filter', 'ucs-streamer'); ?></button>
            </form>
            <div class="ucs-channel-grid">
                <?php if ($query->have_posts()) : while ($query->have_posts()) : $query->the_post(); 
                    $logo = get_post_meta(get_the_ID(), '_ucs_logo_url', true);
                ?>
                    <a href="<?php the_permalink(); ?>" class="ucs-channel-card">
                        <div class="ucs-card-thumb">
                            <?php if($logo): ?><img src="<?php echo esc_url($logo); ?>"><?php else: ?><div class="ucs-no-logo"><span>UCS</span></div><?php endif; ?>
                        </div>
                        <h3><?php the_title(); ?></h3>
                    </a>
                <?php endwhile; else: ?><p class="ucs-no-results">No channels found.</p><?php endif; wp_reset_postdata(); ?>
            </div>
            <div class="ucs-pagination"><?php echo paginate_links(array('total' => $query->max_num_pages)); ?></div>
        </div>
        <style>
            .ucs-directory-controls { display: flex; gap: 10px; margin-bottom: 20px; flex-wrap: wrap; align-items: center; }
            .ucs-directory-controls input, .ucs-directory-controls select { padding: 8px; border: 1px solid #ccc; border-radius: 4px; }
            .ucs-channel-grid { display: grid; grid-template-columns: repeat(auto-fill, minmax(160px, 1fr)); gap: 20px; }
            .ucs-channel-card { border: 1px solid #eee; border-radius: 8px; overflow: hidden; text-decoration: none; color: inherit; display: block; transition: transform 0.2s, box-shadow 0.2s; background: #fff; }
            .ucs-channel-card:hover { transform: translateY(-3px); box-shadow: 0 5px 15px rgba(0,0,0,0.1); }
            .ucs-card-thumb { height: 120px; background: #f9f9f9; display: flex; align-items: center; justify-content: center; overflow: hidden; }
            .ucs-card-thumb img { max-height: 90%; max-width: 90%; object-fit: contain; }
            .ucs-channel-card h3 { padding: 12px; font-size: 15px; margin: 0; text-align: center; border-top: 1px solid #eee; white-space: nowrap; overflow: hidden; text-overflow: ellipsis; }
            .ucs-pagination { margin-top: 30px; text-align: center; }
            .ucs-pagination a, .ucs-pagination span { display: inline-block; padding: 5px 10px; margin: 0 2px; border: 1px solid #ddd; border-radius: 4px; text-decoration: none; }
            .ucs-pagination .current { background: #0073aa; color: #fff; border-color: #0073aa; }
        </style>
        <?php return ob_get_clean();
    }

	public static function get_player_html( $post_id ) {
		$stream_url = get_post_meta( $post_id, '_ucs_stream_url', true );
		if ( empty( $stream_url ) ) return '<div class="ucs-error">Error: No stream URL found.</div>';

        // Config
        $options = get_option( 'ucs_options', array() );
        $access_mode = $options['access_mode'] ?? 'open';
        $product_id = $options['access_product_id'] ?? 0;
        $preview_time = $options['access_preview_time'] ?? 0;
        
        // Access Control
        if ( $access_mode === 'login' && ! is_user_logged_in() ) return self::render_gate( 'login' );
        if ( $access_mode === 'paid' ) {
            $has_purchased = false;
            if ( is_user_logged_in() && function_exists( 'wc_customer_bought_product' ) && ! empty( $product_id ) ) {
                $current_user = wp_get_current_user();
                if ( wc_customer_bought_product( $current_user->user_email, $current_user->ID, $product_id ) ) $has_purchased = true;
            }
            if ( ! $has_purchased ) return self::render_gate( 'paid', $product_id );
        }

        // --- AD PREPARATION ---
        $ad_playlist = []; $top_ads = []; $bottom_ads = [];
        
        if ( class_exists('UCS_License') && UCS_License::is_license_active() ) {
            $fetch_ads = function($ad_ids, $post_id) {
                $ads = [];
                if(empty($ad_ids)) return $ads;
                $channel_cats = wp_get_post_terms( $post_id, 'channel_category', ['fields' => 'slugs'] );
                $channel_countries = wp_get_post_terms( $post_id, 'channel_country', ['fields' => 'slugs'] );

                foreach($ad_ids as $ad_id) {
                    $ad_post = get_post($ad_id);
                    if(!$ad_post || $ad_post->post_status !== 'publish') continue;

                    // Validate Content
                    $type = get_post_meta( $ad_id, '_ucs_ad_type', true );
                    $url = get_post_meta( $ad_id, '_ucs_ad_url', true );
                    
                    // Skip if media is missing
                    if ( empty( $url ) ) continue;

                    $target_cats = get_post_meta( $ad_id, '_ucs_target_cats', true );
                    $target_countries = get_post_meta( $ad_id, '_ucs_target_countries', true );
                    $cat_match = empty( $target_cats ) || ( ! empty( $channel_cats ) && array_intersect( $channel_cats, (array)$target_cats ) );
                    $country_match = empty( $target_countries ) || ( ! empty( $channel_countries ) && array_intersect( $channel_countries, (array)$target_countries ) );

                    if($cat_match && $country_match) {
                        $ads[] = [
                            'id'   => $ad_id,
                            'type' => $type,
                            'url'  => $url,
                            'link' => get_post_meta( $ad_id, '_ucs_ad_link', true ),
                            'skip' => get_post_meta( $ad_id, '_ucs_ad_skip', true ),
                        ];
                    }
                }
                return $ads;
            };

            $all_ads_query = new WP_Query(['post_type'=>'ucs_ad_creative', 'post_status'=>'publish', 'posts_per_page'=>20, 'fields'=>'ids']);
            $ad_playlist = $fetch_ads($all_ads_query->posts, $post_id);

            $top_slot_ids = isset($options['ad_top_player_slot']) ? (array)$options['ad_top_player_slot'] : [];
            $bottom_slot_ids = isset($options['ad_bottom_player_slot']) ? (array)$options['ad_bottom_player_slot'] : [];
            
            $top_ads = $fetch_ads($top_slot_ids, $post_id);
            $bottom_ads = $fetch_ads($bottom_slot_ids, $post_id);
        }

		wp_enqueue_script( 'ucs-hls-js' );
		wp_enqueue_script( 'ucs-player-js' );
		wp_enqueue_style( 'ucs-player-css' );

        $player_id = 'ucs-player-' . absint( $post_id ) . '-' . wp_rand( 100, 999 );
        
        $player_config = array(
			'streamUrl' => esc_url( $stream_url ),
			'autoplay'  => (bool) ($options['player_autoplay'] ?? false),
            'timeLimit' => ( $access_mode === 'timed' && ! is_user_logged_in() ) ? (int)($options['access_preview_time'] ?? 60) : 0,
            'ads'       => $ad_playlist,
            'topAds'    => $top_ads,
            'bottomAds' => $bottom_ads,
            'adConfig'  => array(
                'rotationFreq' => absint($options['ad_rotation_frequency'] ?? 30),
                'midrollInterval' => absint($options['ad_midroll_interval'] ?? 0) * 60,
                'viewFrequency'   => absint($options['ad_view_frequency'] ?? 1)
            )
		);

		ob_start();
		?>
        <div class="ucs-stream-container">
            <div id="ucs-top-slot-<?php echo esc_attr($player_id); ?>" class="ucs-ad-slot ucs-ad-top" style="display:none;"></div>
            <div class="ucs-player-wrapper" id="wrapper-<?php echo esc_attr( $player_id ); ?>">
                <video id="<?php echo esc_attr( $player_id ); ?>" class="ucs-video-js" controls playsinline style="width:100%; height:100%;"></video>
                <div id="ad-overlay-<?php echo esc_attr( $player_id ); ?>" class="ucs-ad-overlay" style="display:none;">
                    <div class="ucs-ad-content">
                        <img id="ad-img-<?php echo esc_attr( $player_id ); ?>" src="" style="display:none;" />
                        <button id="ad-skip-<?php echo esc_attr( $player_id ); ?>" class="ucs-ad-skip-btn">Skip</button>
                    </div>
                </div>
                <div id="gate-<?php echo esc_attr( $player_id ); ?>" class="ucs-gate-overlay" style="display:none;">
                    <h3><?php _e('Preview Ended', 'ucs-streamer'); ?></h3>
                    <a href="<?php echo wp_login_url( get_permalink() ); ?>" class="button"><?php _e('Log In', 'ucs-streamer'); ?></a>
                </div>
            </div>
            <div id="ucs-bottom-slot-<?php echo esc_attr($player_id); ?>" class="ucs-ad-slot ucs-ad-bottom" style="display:none;"></div>
            <?php self::render_related_channels($post_id); ?>
			<script type="text/javascript">
				(function() {
					if (typeof ucsPlayerQueue === 'undefined') { window.ucsPlayerQueue = []; }
					window.ucsPlayerQueue.push({
						id: '<?php echo esc_attr( $player_id ); ?>',
						config: <?php echo wp_json_encode( $player_config ); ?>
					});
				})();
			</script>
        </div>
		<?php
		return ob_get_clean();
	}

    private static function render_related_channels($post_id) {
        $cats = wp_get_post_terms($post_id, 'channel_category', ['fields'=>'ids']);
        if(empty($cats)) return;
        $related = new WP_Query([ 'post_type' => 'channel', 'post_status' => 'publish', 'posts_per_page' => 4, 'post__not_in' => [$post_id], 'tax_query' => [[ 'taxonomy' => 'channel_category', 'field' => 'term_id', 'terms' => $cats ]] ]);
        if($related->have_posts()): ?>
            <div class="ucs-related-channels">
                <h3><?php _e('Related Channels', 'ucs-streamer'); ?></h3>
                <div class="ucs-related-grid">
                    <?php while($related->have_posts()): $related->the_post(); $logo = get_post_meta(get_the_ID(), '_ucs_logo_url', true); ?>
                        <a href="<?php the_permalink(); ?>" class="ucs-related-card"><?php if($logo): ?><img src="<?php echo esc_url($logo); ?>"><?php else: ?><span><?php the_title(); ?></span><?php endif; ?></a>
                    <?php endwhile; ?>
                </div>
            </div>
        <?php endif; wp_reset_postdata();
    }

    private static function render_gate( $type, $product_id = 0 ) {
        $msg = $type === 'login' ? 'Login Required' : 'Premium Only';
        $btn = $type === 'login' ? 'Log In' : 'Purchase Access';
        $link = $type === 'login' ? wp_login_url() : get_permalink($product_id);
        return '<div class="ucs-gate-static"><h3>'.$msg.'</h3><a href="'.$link.'" class="button">'.$btn.'</a></div>';
    }
}