<?php
class UCS_Importer {
    
    /**
     * Handle CSV Upload (LOCKED)
     */
    public static function handle_csv_upload() {
        // 1. Security & Permissions
        if ( ! current_user_can('manage_options') ) return;
        check_admin_referer( 'ucs_csv_upload_nonce', 'ucs_csv_nonce' ); // Standard WordPress security check
        
        // 2. *** LICENSE LOCK ***
        if ( ! class_exists('UCS_License') || ! UCS_License::is_valid() ) {
            // Stop execution if no license
            wp_die( 
                '<h1>Feature Locked</h1><p>You must activate a valid license key to use the CSV Importer.</p><p><a href="admin.php?page=ucs-settings&tab=license">Go to License Settings</a></p>', 
                'License Required', 
                ['response' => 403] 
            );
        }

        // 3. Process File
        if ( isset($_FILES['ucs_csv_file']) && $_FILES['ucs_csv_file']['error'] === UPLOAD_ERR_OK ) {
            $file = $_FILES['ucs_csv_file']['tmp_name'];
            $handle = fopen($file, 'r');
            if ($handle) {
                fgetcsv($handle); // Skip header row
                while ( ($row = fgetcsv($handle)) !== false ) {
                    // Expected CSV: Name, Logo, URL, Category, Country
                    // Map: 0=name, 1=logo, 2=url, 3=cat, 4=country
                    if( count($row) >= 3 ) { // Basic validation
                        self::create_channel($row[0], $row[2], $row[1], $row[3] ?? '', $row[4] ?? '');
                    }
                }
                fclose($handle);
            }
        }
        
        // Redirect back to settings
        wp_redirect( admin_url('admin.php?page=ucs-settings') ); 
        exit;
    }

    /**
     * Handle M3U AJAX Import (LOCKED)
     */
    public static function handle_m3u_import_ajax() {
        check_ajax_referer('ucs_import_m3u_nonce', 'security');
        
        // *** LICENSE LOCK ***
        if ( ! UCS_License::is_valid() ) {
            wp_send_json_error( 'License required to use Auto-Import.' );
            return;
        }

        $url = 'https://iptv-org.github.io/iptv/index.m3u';
        $response = wp_remote_get($url, ['timeout'=>60, 'sslverify'=>false]);
        
        if(is_wp_error($response)) {
            wp_send_json_error('Fetch failed: ' . $response->get_error_message());
            return;
        }
        
        $lines = explode("\n", wp_remote_retrieve_body($response));
        $count = 0; 
        $buffer = [];
        
        foreach($lines as $line) {
            $line = trim($line);
            if(strpos($line, '#EXTINF') === 0) {
                preg_match('/tvg-logo="([^"]*)"/', $line, $logo);
                $parts = explode(',', $line);
                $buffer['name'] = end($parts);
                $buffer['logo'] = $logo[1] ?? '';
            } elseif(strpos($line, 'http') === 0 && !empty($buffer)) {
                self::create_channel($buffer['name'], $line, $buffer['logo']);
                $count++;
                $buffer = [];
                if($count >= 50) break; // Limit batch size for performance
            }
        }
        wp_send_json_success(['count'=>$count]);
    }

    private static function create_channel($name, $url, $logo='', $cat='', $country='') {
        $name = sanitize_text_field($name);
        if( empty($name) || empty($url) ) return;

        // Check duplicates
        $exists = get_page_by_title($name, OBJECT, 'ucs_channel');
        if($exists) return;
        
        $pid = wp_insert_post(['post_title'=>$name, 'post_type'=>'ucs_channel', 'post_status'=>'publish']);
        update_post_meta($pid, 'ucs_m3u8_link', esc_url_raw($url));
        update_post_meta($pid, 'ucs_logo_url', esc_url_raw($logo));
        
        if($cat) wp_set_object_terms($pid, sanitize_text_field($cat), 'ucs_category');
        if($country) wp_set_object_terms($pid, sanitize_text_field($country), 'ucs_country');
    }
}