<?php
class UCS_License {

    /**
     * Check if license is active locally.
     */
    public static function is_valid() {
        return ( get_option( 'ucs_license_status' ) === 'valid' );
    }

    /**
     * Retrieve the saved key.
     */
    public static function get_key() {
        return get_option( 'ucs_license_key', '' );
    }

    /**
     * Connect to the remote server to validate the key.
     */
    public static function activate_license( $license_key ) {
        
        $license_key = sanitize_text_field( trim( $license_key ) );
        $master_key  = '0D144-052A5-9BEAA-AE30B'; // Your hardcoded key

        // --- 1. MASTER KEY BYPASS (Prevents Loopback Crash) ---
        // If the key matches your master key, validate instantly without API call.
        if ( $license_key === $master_key ) {
            update_option( 'ucs_license_status', 'valid' );
            update_option( 'ucs_license_key', $license_key );
            return true;
        }

        // --- 2. REMOTE API CHECK (For your customers) ---
        // We accept both GET and POST depending on server config, trying GET first.
        $api_params = array(
            'license_key' => $license_key,
            'product_id'  => UCS_PREMIUM_PRODUCT_ID,
            'url'         => home_url(), 
        );

        $request_url = add_query_arg( $api_params, UCS_UWLM_API_ENDPOINT );

        $response = wp_remote_get( $request_url, array(
            'timeout'   => 15,
            'sslverify' => false, 
            'headers'   => array( 'Accept' => 'application/json' )
        ) );

        if ( is_wp_error( $response ) ) {
            return new WP_Error( 'connection_failed', 'Connection error: ' . $response->get_error_message() );
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $body_response = json_decode( wp_remote_retrieve_body( $response ), true );

        $is_success = false;
        
        // Check standard JSON success responses
        if ( $response_code === 200 ) {
            if ( isset( $body_response['success'] ) && $body_response['success'] == true ) {
                $is_success = true;
            } elseif ( isset( $body_response['status'] ) && in_array( $body_response['status'], ['active', 'success', 'valid'] ) ) {
                $is_success = true;
            }
        }

        if ( $is_success ) {
            update_option( 'ucs_license_status', 'valid' );
            update_option( 'ucs_license_key', $license_key );
            return true;
        } else {
            $msg = isset( $body_response['message'] ) ? $body_response['message'] : 'License invalid.';
            if ( $response_code === 404 ) $msg = 'License Server Endpoint not found.';
            
            self::deactivate_local();
            return new WP_Error( 'invalid_license', $msg );
        }
    }

    /**
     * Clear local license data.
     */
    public static function deactivate_local() {
        delete_option( 'ucs_license_status' );
        delete_option( 'ucs_license_key' );
    }

    /**
     * Handle Admin Form Submission
     */
    public static function handle_activation_request() {
        if ( ! isset( $_POST['ucs_license_nonce'] ) || ! wp_verify_nonce( $_POST['ucs_license_nonce'], 'ucs_activate_license' ) ) {
            return;
        }

        // Activate
        if ( isset( $_POST['ucs_activate_btn'] ) && ! empty( $_POST['ucs_license_key'] ) ) {
            $result = self::activate_license( $_POST['ucs_license_key'] );
            if ( is_wp_error( $result ) ) {
                add_settings_error( 'ucs_license_messages', 'ucs_activation_error', $result->get_error_message(), 'error' );
            } else {
                add_settings_error( 'ucs_license_messages', 'ucs_activation_success', 'License Activated Successfully!', 'updated' );
            }
        }

        // Deactivate
        if ( isset( $_POST['ucs_deactivate_btn'] ) ) {
            self::deactivate_local();
            add_settings_error( 'ucs_license_messages', 'ucs_deactivation_success', 'License Deactivated.', 'updated' );
        }
    }
}