<?php
class UCS_Player {

    /**
     * Renders the Channel Directory (Shortcode: [ucs_channel_directory])
     * Uses AJAX to load and filter channels to improve performance.
     */
    public static function render_directory( $atts ) {
        ob_start();
        ?>
        <div class="ucs-directory-wrapper">
            <h2><?php esc_html_e( 'Live Channel Directory', 'ucs' ); ?></h2>
            
            <form id="ucs-filter-form" class="ucs-filter-form">
                <div class="ucs-filter-group">
                    <label><?php esc_html_e( 'Category', 'ucs' ); ?></label>
                    <?php 
                    wp_dropdown_categories( array(
                        'show_option_all' => __( 'All Categories', 'ucs' ), 
                        'taxonomy' => 'ucs_category', 
                        'name' => 'category_filter', 
                        'id' => 'category_filter', 
                        'hierarchical' => true, 
                        'value_field' => 'slug'
                    ) ); 
                    ?>
                </div>
                <div class="ucs-filter-group">
                    <label><?php esc_html_e( 'Country', 'ucs' ); ?></label>
                    <?php 
                    wp_dropdown_categories( array(
                        'show_option_all' => __( 'All Countries', 'ucs' ), 
                        'taxonomy' => 'ucs_country', 
                        'name' => 'country_filter', 
                        'id' => 'country_filter', 
                        'hierarchical' => false, 
                        'value_field' => 'slug'
                    ) ); 
                    ?>
                </div>
                <div class="ucs-filter-group">
                    <button type="submit" class="button button-primary"><?php esc_html_e( 'Filter', 'ucs' ); ?></button>
                </div>
            </form>
            
            <div id="ucs-directory-grid" class="ucs-channel-grid-loading">
                <p><?php esc_html_e( 'Loading channels...', 'ucs' ); ?></p>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * AJAX Handler for fetching and filtering channels.
     */
    public static function handle_load_channels_ajax() {
        check_ajax_referer( 'ucs_load_channels_nonce', 'security' );

        $category = isset( $_POST['category'] ) ? sanitize_text_field( $_POST['category'] ) : '';
        $country = isset( $_POST['country'] ) ? sanitize_text_field( $_POST['country'] ) : '';

        $query_args = array(
            'post_type'      => 'ucs_channel', 
            'posts_per_page' => 48, 
            'post_status'    => 'publish',
            'tax_query'      => array( 'relation' => 'AND' ),
            // Performance optimizations
            'no_found_rows'          => true, 
            'update_post_term_cache' => false, 
            'update_post_meta_cache' => true,
        );

        if ( ! empty( $category ) ) {
            $query_args['tax_query'][] = array(
                'taxonomy' => 'ucs_category', 
                'field' => 'slug', 
                'terms' => $category
            );
        }
        if ( ! empty( $country ) ) {
            $query_args['tax_query'][] = array(
                'taxonomy' => 'ucs_country', 
                'field' => 'slug', 
                'terms' => $country
            );
        }

        $channels_query = new WP_Query( $query_args );
        
        ob_start();
        if ( $channels_query->have_posts() ) {
            echo '<ul class="ucs-channel-grid">';
            while ( $channels_query->have_posts() ) {
                $channels_query->the_post();
                $logo_url = get_post_meta( get_the_ID(), 'ucs_logo_url', true );
                
                // Lazy loading image tag
                $logo = $logo_url 
                    ? '<img loading="lazy" src="' . esc_url( $logo_url ) . '" alt="' . esc_attr( get_the_title() ) . '" class="ucs-channel-logo">' 
                    : '<div class="ucs-logo-placeholder"></div>';
                
                echo '<li class="ucs-channel-item">';
                echo '<a href="' . esc_url( get_permalink() ) . '" title="' . esc_attr( get_the_title() ) . '">';
                echo $logo; 
                echo '<span>' . esc_html( get_the_title() ) . '</span>';
                echo '</a></li>';
            }
            echo '</ul>';
        } else {
            echo UCS_Utils::generate_message_box( __( 'No channels found matching your criteria.', 'ucs' ), 'info' );
        }
        wp_reset_postdata();
        $html = ob_get_clean();

        wp_send_json_success( $html );
    }

    /**
     * Renders the main Video.js Player (Shortcode: [ucs_player])
     */
    public static function render_player( $atts ) {
        $atts = shortcode_atts( array( 'id' => get_the_ID() ), $atts, 'ucs_player' );
        $channel_id = absint( $atts['id'] );
        
        // Validation
        if ( ! $channel_id || get_post_type( $channel_id ) !== 'ucs_channel' ) {
            return UCS_Utils::generate_message_box( __( 'Invalid channel ID.', 'ucs' ), 'error' );
        }

        // Get Data
        $m3u8_url = get_post_meta( $channel_id, 'ucs_m3u8_link', true );
        $logo_url = get_post_meta( $channel_id, 'ucs_logo_url', true );
        
        // Get Options
        $ucs_options = get_option( 'ucs_settings', array() );
        $access_mode = $ucs_options['access_mode'] ?? 'open';
        $product_id  = absint( $ucs_options['woocommerce_product_id'] ?? 0 );
        
        // Check Access
        $has_paid_access = UCS_Utils::has_user_purchased_access( $product_id );
        $is_locked = false;

        // "Paid Hard Gating" Logic
        if ( $access_mode === 'paid_hard' && ! $has_paid_access ) {
            $is_locked = true;
        }

        ob_start();
        ?>
        <div class="ucs-player-flex-container">
            <div class="ucs-video-player-area">
                <?php if ( empty( $m3u8_url ) ) : ?>
                    <?php echo UCS_Utils::generate_message_box( __( 'Stream URL is missing for this channel.', 'ucs' ), 'error' ); ?>
                <?php else : ?>
                    
                    <video id="ucs-player-<?php echo esc_attr( $channel_id ); ?>" 
                           class="video-js vjs-default-skin ucs-hls-player" 
                           controls 
                           preload="auto" 
                           data-stream-url="<?php echo esc_url( $m3u8_url ); ?>" 
                           data-access-mode="<?php echo esc_attr( $access_mode ); ?>" 
                           data-paid-access="<?php echo $has_paid_access ? 'true' : 'false'; ?>"
                           <?php if ($logo_url) echo 'poster="' . esc_url($logo_url) . '"'; ?>>
                        <p class="vjs-no-js"><?php esc_html_e( 'To view this video please enable JavaScript.', 'ucs' ); ?></p>
                    </video>
                    
                    <div id="ucs-overlay-<?php echo esc_attr( $channel_id ); ?>" class="ucs-paywall-overlay" style="display:none;">
                        <?php if ( $is_locked ) : ?>
                            <h2><?php esc_html_e('Premium Content Locked', 'ucs'); ?></h2>
                            <p><?php esc_html_e('Purchase access to continue watching.', 'ucs'); ?></p>
                            <?php if ( $product_id ) : ?>
                                <a href="<?php echo esc_url( get_permalink( $product_id ) ); ?>" class="ucs-purchase-btn button button-primary">
                                    <?php esc_html_e('Unlock Access Now', 'ucs'); ?>
                                </a>
                            <?php endif; ?>
                        <?php elseif ( ($access_mode === 'login' || $access_mode === 'timed_soft') && ! is_user_logged_in() ) : ?>
                            <h2><?php esc_html_e('Login Required', 'ucs'); ?></h2>
                            <p><?php esc_html_e('Log in to view this channel.', 'ucs'); ?></p>
                            <a href="<?php echo esc_url( wp_login_url( get_permalink() ) ); ?>" class="ucs-login-btn button button-primary">
                                <?php esc_html_e('Log In', 'ucs'); ?>
                            </a>
                        <?php endif; ?>
                    </div>
                    
                <?php endif; ?>
                
                <?php 
                $show_embed = $ucs_options['embed_code_toggle'] ?? '0';
                if ( $show_embed === '1' ) : 
                ?>
                    <div class="ucs-embed-container">
                        <button class="ucs-get-embed-btn button" data-channel-id="<?php echo esc_attr( $channel_id ); ?>">
                            <?php esc_html_e( 'Get Embed Code', 'ucs' ); ?>
                        </button>
                    </div>
                <?php endif; ?>
            </div>

            <div class="ucs-engagement-sidebar">
                <?php if ( is_user_logged_in() ) : ?>
                    <h3><?php esc_html_e('Live Chat', 'ucs'); ?></h3>
                    <div class="ucs-chat-box">
                        <p><?php esc_html_e('Welcome! Join the conversation...', 'ucs'); ?></p>
                    </div>
                <?php else : ?>
                    <?php echo UCS_Utils::generate_message_box( __( 'Log in to participate in the live chat.', 'ucs' ) ); ?>
                <?php endif; ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
}