<?php
/**
 * Registers the 'channel' Custom Post Type, Taxonomies, and Meta Boxes.
 *
 * @package UCS_Streamer
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class UCS_Post_Types {

	/**
	 * Initialize hooks
	 */
	public static function init() {
		// CPT and Taxonomy registration MUST happen on 'init'.
		add_action( 'init', array( __CLASS__, 'register_post_type' ) );
		add_action( 'init', array( __CLASS__, 'register_taxonomies' ) );
		add_action( 'add_meta_boxes', array( __CLASS__, 'add_meta_boxes' ) );
		add_action( 'save_post_channel', array( __CLASS__, 'save_meta_boxes' ) );
	}

	/**
	 * Register the 'channel' CPT.
	 */
	public static function register_post_type() {
		$labels = array(
			'name'                  => _x( 'Channels', 'Post Type General Name', 'ucs-streamer' ),
			'singular_name'         => _x( 'Channel', 'Post Type Singular Name', 'ucs-streamer' ),
			'menu_name'             => __( 'UCS Streamer', 'ucs-streamer' ),
			'name_admin_bar'        => __( 'Channel', 'ucs-streamer' ),
			'archives'              => __( 'Channel Archives', 'ucs-streamer' ),
			'attributes'            => __( 'Channel Attributes', 'ucs-streamer' ),
			'parent_item_colon'     => __( 'Parent Channel:', 'ucs-streamer' ),
			'all_items'             => __( 'All Channels', 'ucs-streamer' ),
			'add_new_item'          => __( 'Add New Channel', 'ucs-streamer' ),
			'add_new'               => __( 'Add New', 'ucs-streamer' ),
			'new_item'              => __( 'New Channel', 'ucs-streamer' ),
			'edit_item'             => __( 'Edit Channel', 'ucs-streamer' ),
			'update_item'           => __( 'Update Channel', 'ucs-streamer' ),
			'view_item'             => __( 'View Channel', 'ucs-streamer' ),
			'view_items'            => __( 'View Channels', 'ucs-streamer' ),
			'search_items'          => __( 'Search Channel', 'ucs-streamer' ),
			'not_found'             => __( 'Not found', 'ucs-streamer' ),
			'not_found_in_trash'    => __( 'Not found in Trash', 'ucs-streamer' ),
			'featured_image'        => __( 'Channel Logo', 'ucs-streamer' ),
			'set_featured_image'    => __( 'Set channel logo', 'ucs-streamer' ),
			'remove_featured_image' => __( 'Remove channel logo', 'ucs-streamer' ),
			'use_featured_image'    => __( 'Use as channel logo', 'ucs-streamer' ),
			'insert_into_item'      => __( 'Insert into channel', 'ucs-streamer' ),
			'uploaded_to_this_item' => __( 'Uploaded to this channel', 'ucs-streamer' ),
			'items_list'            => __( 'Channels list', 'ucs-streamer' ),
			'items_list_navigation' => __( 'Channels list navigation', 'ucs-streamer' ),
			'filter_items_list'     => __( 'Filter channels list', 'ucs-streamer' ),
		);
		$args = array(
			'label'                 => __( 'Channel', 'ucs-streamer' ),
			'description'           => __( 'A post type for live streaming channels.', 'ucs-streamer' ),
			'labels'                => $labels,
			'supports'              => array( 'title', 'editor', 'thumbnail', 'revisions' ),
			'taxonomies'            => array( 'channel_category', 'channel_country' ),
			'hierarchical'          => false,
			'public'                => true,
			'show_ui'               => true,
			'show_in_menu'          => true,
			'menu_position'         => 20,
			'menu_icon'             => 'dashicons-video-alt3',
			'show_in_admin_bar'     => true,
			'show_in_nav_menus'     => true,
			'can_export'            => true,
			'has_archive'           => 'streams', // Archive page at /streams/
			'exclude_from_search'   => false,
			'publicly_queryable'    => true,
			'capability_type'       => 'post',
			'show_in_rest'          => true,
			'rewrite'               => array( 'slug' => 'stream', 'with_front' => false ), // Single page at /stream/channel-name/
		);
		register_post_type( 'channel', $args );
	}

	/**
	 * Register 'channel_category' and 'channel_country' taxonomies.
	 */
	public static function register_taxonomies() {
		// Category Taxonomy
		$cat_labels = array(
			'name'              => _x( 'Categories', 'taxonomy general name', 'ucs-streamer' ),
			'singular_name'     => _x( 'Category', 'taxonomy singular name', 'ucs-streamer' ),
			'search_items'      => __( 'Search Categories', 'ucs-streamer' ),
			'all_items'         => __( 'All Categories', 'ucs-streamer' ),
			'parent_item'       => __( 'Parent Category', 'ucs-streamer' ),
			'parent_item_colon' => __( 'Parent Category:', 'ucs-streamer' ),
			'edit_item'         => __( 'Edit Category', 'ucs-streamer' ),
			'update_item'       => __( 'Update Category', 'ucs-streamer' ),
			'add_new_item'      => __( 'Add New Category', 'ucs-streamer' ),
			'new_item_name'     => __( 'New Category Name', 'ucs-streamer' ),
			'menu_name'         => __( 'Categories', 'ucs-streamer' ),
		);
		$cat_args = array(
			'hierarchical'      => true,
			'labels'            => $cat_labels,
			'show_ui'           => true,
			'show_admin_column' => true,
			'query_var'         => true,
			'rewrite'           => array( 'slug' => 'channel-category' ),
			'show_in_rest'      => true,
		);
		register_taxonomy( 'channel_category', array( 'channel' ), $cat_args );

		// Country Taxonomy
		$country_labels = array(
			'name'              => _x( 'Countries', 'taxonomy general name', 'ucs-streamer' ),
			'singular_name'     => _x( 'Country', 'taxonomy singular name', 'ucs-streamer' ),
			'search_items'      => __( 'Search Countries', 'ucs-streamer' ),
			'all_items'         => __( 'All Countries', 'ucs-streamer' ),
			'parent_item'       => __( 'Parent Country', 'ucs-streamer' ),
			'parent_item_colon' => __( 'Parent Country:', 'ucs-streamer' ),
			'edit_item'         => __( 'Edit Country', 'ucs-streamer' ),
			'update_item'       => __( 'Update Country', 'ucs-streamer' ),
			'add_new_item'      => __( 'Add New Country', 'ucs-streamer' ),
			'new_item_name'     => __( 'New Country Name', 'ucs-streamer' ),
			'menu_name'         => __( 'Countries', 'ucs-streamer' ),
		);
		$country_args = array(
			'hierarchical'      => false, // Countries are flat
			'labels'            => $country_labels,
			'show_ui'           => true,
			'show_admin_column' => true,
			'query_var'         => true,
			'rewrite'           => array( 'slug' => 'channel-country' ),
			'show_in_rest'      => true,
		);
		register_taxonomy( 'channel_country', array( 'channel' ), $country_args );
	}

	/**
	 * Add meta box to 'channel' CPT.
	 */
	public static function add_meta_boxes() {
		add_meta_box(
			'ucs_channel_details',
			__( 'Channel Details', 'ucs-streamer' ),
			array( __CLASS__, 'render_meta_box' ),
			'channel',
			'normal',
			'high'
		);
	}

	/**
	 * Render the meta box content.
	 */
	public static function render_meta_box( $post ) {
		// Add a nonce field
		wp_nonce_field( 'ucs_save_channel_details', 'ucs_channel_details_nonce' );

		// Get existing values
		$stream_url = get_post_meta( $post->ID, '_ucs_stream_url', true );
		$logo_url = get_post_meta( $post->ID, '_ucs_logo_url', true );

		// Stream URL
		echo '<p>';
		echo '<label for="ucs_stream_url"><strong>' . __( 'Stream URL (.m3u8, .mpd, etc.)', 'ucs-streamer' ) . '</strong></label>';
		echo '<input type="url" id="ucs_stream_url" name="ucs_stream_url" value="' . esc_url( $stream_url ) . '" class="widefat" placeholder="https://example.com/stream.m3u8" required />';
		echo '</p>';

		// Logo URL
		echo '<p>';
		echo '<label for="ucs_logo_url"><strong>' . __( 'Logo URL (Optional)', 'ucs-streamer' ) . '</strong></label>';
		echo '<input type="url" id="ucs_logo_url" name="ucs_logo_url" value="' . esc_url( $logo_url ) . '" class="widefat" placeholder="https://example.com/logo.png" />';
		echo '<span class="description">' . __( 'Optional. Overrides the "Featured Image". Use this for external logo URLs from M3U/CSV imports.', 'ucs-streamer' ) . '</span>';
		echo '</p>';
	}

	/**
	 * Save meta box data.
	 */
	public static function save_meta_boxes( $post_id ) {
		// Check nonce
		if ( ! isset( $_POST['ucs_channel_details_nonce'] ) || ! wp_verify_nonce( $_POST['ucs_channel_details_nonce'], 'ucs_save_channel_details' ) ) {
			return;
		}

		// Check if this is an autosave
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		// Check permissions
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		// Save Stream URL
		if ( isset( $_POST['ucs_stream_url'] ) ) {
			update_post_meta( $post_id, '_ucs_stream_url', esc_url_raw( $_POST['ucs_stream_url'] ) );
		}

		// Save Logo URL
		if ( isset( $_POST['ucs_logo_url'] ) ) {
			update_post_meta( $post_id, '_ucs_logo_url', esc_url_raw( $_POST['ucs_logo_url'] ) );
		}
	}
}