<?php
/**
 * Handles the one-time import of default channels.
 * Optimized for large datasets and various CSV formats.
 *
 * @package UCS_Streamer
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class UCS_Preloader {

	public static function init() {
		add_action( 'ucs_run_channel_preload', array( __CLASS__, 'run_preload' ) );
	}

	public static function schedule_preload() {
		if ( ! get_option( '_ucs_channels_preloaded' ) ) {
			if ( ! wp_next_scheduled( 'ucs_run_channel_preload' ) ) {
				wp_schedule_single_event( time() + 10, 'ucs_run_channel_preload' );
			}
		}
	}

	/**
	 * Main execution function.
	 * Can be triggered by cron or manually.
	 */
	public static function run_preload() {
		// Increase limits for large imports (10k+ channels)
		if ( function_exists( 'set_time_limit' ) ) {
			set_time_limit( 0 ); // Unlimited time
		}
		if ( function_exists( 'ini_set' ) ) {
			ini_set( 'memory_limit', '512M' );
			ini_set( 'auto_detect_line_endings', true ); // Crucial for CSVs
		}

		if ( ! class_exists( 'UCS_Importer' ) ) {
			$importer_path = UCS_PLUGIN_DIR . 'includes/class-ucs-importer.php';
			if ( file_exists( $importer_path ) ) {
				require_once $importer_path;
			} else {
				return;
			}
		}

		// File mapping
		$files = array(
			'channels3.csv' => 'csv_format_3', // name,logoUrl,streamUrl...
			'channels2.csv' => 'csv_format_2', // Group Title,Channel Name...
			'channel4.csv'  => 'csv_format_4', // Channel Name,Logo,Stream URL (with headers)
		);

		foreach ( $files as $file => $format ) {
			$filepath = UCS_PLUGIN_DIR . 'data/' . $file;
			if ( file_exists( $filepath ) ) {
				self::import_file( $filepath, $format );
			}
		}
		
		// Mark as complete so it doesn't auto-run again
		update_option( '_ucs_channels_preloaded', true, 'no' );
	}

	/**
	 * Logic to read specific file formats
	 */
	private static function import_file( $filepath, $format ) {
		$handle = fopen( $filepath, 'r' );
		if ( ! $handle ) {
			return;
		}

		// Skip headers based on known formats
		if ( $format === 'csv_format_3' ) {
			fgetcsv( $handle ); 
		} elseif ( $format === 'csv_format_2' ) {
			fgetcsv( $handle ); 
		} elseif ( $format === 'csv_format_4' ) {
			// channel4.csv has ~8 header lines
			for ( $i = 0; $i < 8; $i++ ) { fgetcsv( $handle ); }
		}

		$current_country = '';
		
		while ( ( $row = fgetcsv( $handle ) ) !== false ) {
			if ( empty( $row ) ) continue;

			$data = array();

			// --- PARSING LOGIC ---
			try {
				if ( $format === 'csv_format_3' ) {
					// Expected: name, logoUrl, streamUrl, category, country
					if ( count( $row ) < 5 ) continue;
					$data = array(
						'name' => trim( $row[0] ),
						'logo' => self::clean_url( trim( $row[1] ) ),
						'url'  => self::clean_url( trim( $row[2] ) ),
						'cat'  => trim( $row[3] ),
						'country' => trim( $row[4] ),
					);
				} elseif ( $format === 'csv_format_2' ) {
					// Expected: Group Title, Channel Name, Logo URL, tvg-id, Stream URL
					if ( count( $row ) < 5 ) continue;
					$data = array(
						'country' => trim( $row[0] ),
						'name' => trim( $row[1] ),
						'logo' => self::clean_url( trim( $row[2] ) ),
						'url'  => self::clean_url( trim( $row[4] ) ),
						'cat'  => '', // Use Country as category or leave blank
					);
				} elseif ( $format === 'csv_format_4' ) {
					// This file has section headers like "Kosovo (4 Channels)"
					if ( ! empty( $row[0] ) && empty( $row[1] ) && empty( $row[2] ) ) {
						// It's a section header
						preg_match( '/^([^(]+)/', $row[0], $matches );
						if ( ! empty( $matches[1] ) ) {
							$current_country = trim( $matches[1] );
						}
						// Skip next 2 lines (blanks/subheaders) usually found in this format
						fgetcsv( $handle ); 
						fgetcsv( $handle ); 
						continue;
					}
					
					if ( count( $row ) < 3 || empty( $row[0] ) || empty( $row[2] ) ) continue;
					
					$data = array(
						'name' => trim( $row[0] ),
						'logo' => trim( $row[1] ), // Sometimes just text "Logo", checked below
						'url'  => self::clean_url( trim( $row[2] ) ),
						'country' => $current_country,
						'cat'  => '',
					);
				}

				// --- VALIDATION & CREATION ---
				if ( ! empty( $data['name'] ) && ! empty( $data['url'] ) ) {
					// Validate URL roughly
					if ( filter_var( $data['url'], FILTER_VALIDATE_URL ) ) {
						// Only pass valid logo URLs
						if ( ! filter_var( $data['logo'], FILTER_VALIDATE_URL ) ) {
							$data['logo'] = '';
						}
						
						UCS_Importer::create_channel( 
							$data['name'], 
							$data['url'], 
							$data['logo'], 
							$data['cat'], 
							$data['country'] 
						);
					}
				}
			} catch ( Exception $e ) {
				continue;
			}
		}
		
		fclose( $handle );
	}

	/**
	 * Robust URL Cleaner
	 * Handles standard URLs and those accidentally wrapped in Google redirects.
	 */
	private static function clean_url( $url ) {
		if ( empty( $url ) ) return '';

		// 1. Handle Google Redirects
		if ( strpos( $url, 'google.com/search?q=' ) !== false ) {
			// Extract the real URL between 'q=' and the next '&'
			$parts = explode( 'q=', $url );
			if ( isset( $parts[1] ) ) {
				$sub_parts = explode( '&', $parts[1] );
				$url = urldecode( $sub_parts[0] );
			}
		}
		
		// 2. Basic cleanup
		return esc_url_raw( trim( $url ) );
	}
}