<?php
class UCS_Settings {
    const SETTINGS_GROUP = 'ucs_settings_group';
    const SETTINGS_PAGE_SLUG = 'ucs-settings';

    public static function add_admin_menu() {
        add_menu_page( 'UCS Settings', 'UCS Streamer', 'manage_options', self::SETTINGS_PAGE_SLUG, array( __CLASS__, 'render_settings_page' ), 'dashicons-video-alt', 50 );
        add_submenu_page( self::SETTINGS_PAGE_SLUG, 'Analytics', 'Analytics', 'manage_options', 'ucs-analytics', array( __CLASS__, 'render_analytics_page' ) );
    }

    public static function register_settings() {
        // Handle License Activation
        if ( isset( $_POST['ucs_license_nonce'] ) ) {
            UCS_License::handle_activation_request();
        }

        register_setting( self::SETTINGS_GROUP, 'ucs_settings', array( __CLASS__, 'sanitize_settings' ) );

        // --- TAB 1: GENERAL (Mixed Free & Premium) ---
        add_settings_section( 'ucs_section_general', 'General Configuration', null, self::SETTINGS_PAGE_SLUG );
        
        // Premium Fields
        add_settings_field( 'access_mode', 'Global Access Mode <span class="dashicons dashicons-lock"></span>', array( __CLASS__, 'render_access_mode_field' ), self::SETTINGS_PAGE_SLUG, 'ucs_section_general' );
        add_settings_field( 'paywall_limit_sec', 'Free Trial Duration <span class="dashicons dashicons-lock"></span>', array( __CLASS__, 'render_paywall_limit_field' ), self::SETTINGS_PAGE_SLUG, 'ucs_section_general' );
        add_settings_field( 'woocommerce_product_id', 'Paywall Product <span class="dashicons dashicons-lock"></span>', array( __CLASS__, 'render_wc_product_field' ), self::SETTINGS_PAGE_SLUG, 'ucs_section_general' );
        
        // Free Fields
        add_settings_field( 'embed_code_toggle', 'Public Embed Button', array( __CLASS__, 'render_embed_toggle_field' ), self::SETTINGS_PAGE_SLUG, 'ucs_section_general' );

        // --- TAB 2: ADVERTISING (Premium Only) ---
        add_settings_section( 'ucs_section_ads', 'Video Advertising (VAST/IMA)', array( __CLASS__, 'section_ads_cb' ), 'ucs-ads-settings' );
        add_settings_field( 'vast_pre_roll', 'Pre-roll VAST URL', array( __CLASS__, 'render_ad_field' ), 'ucs-ads-settings', 'ucs_section_ads', ['key' => 'vast_pre_roll'] );
        add_settings_field( 'vast_mid_roll', 'Mid-roll VAST URL', array( __CLASS__, 'render_ad_field' ), 'ucs-ads-settings', 'ucs_section_ads', ['key' => 'vast_mid_roll'] );
        add_settings_field( 'vast_mid_roll_interval', 'Mid-roll Interval (Min)', array( __CLASS__, 'render_ad_interval_field' ), 'ucs-ads-settings', 'ucs_section_ads', ['key' => 'vast_mid_roll_interval'] );

        // --- TAB 3: DATA (Import) ---
        add_settings_section( 'ucs_section_data', 'Channel Data', null, 'ucs-data-settings' );
        add_settings_field( 'data_tools', 'Auto Import', array( __CLASS__, 'render_import_tools' ), 'ucs-data-settings', 'ucs_section_data' );
        
        // --- TAB 4: LICENSE ---
        add_settings_section( 'ucs_section_license', 'License Activation', array( __CLASS__, 'section_license_cb' ), 'ucs-license-settings' );
        add_settings_field( 'license_key', 'License Key', array( __CLASS__, 'render_license_field' ), 'ucs-license-settings', 'ucs_section_license' );
    }

    public static function sanitize_settings( $input ) {
        // If license is invalid, DO NOT save Premium fields (Prevent tampering via DevTools)
        $is_valid = UCS_License::is_valid();
        $existing = get_option( 'ucs_settings', [] );

        $output = $existing;

        // Always save these (Free features)
        $output['embed_code_toggle'] = isset( $input['embed_code_toggle'] ) ? '1' : '0';

        // Only save these if Valid License
        if ( $is_valid ) {
            $output['access_mode'] = sanitize_text_field( $input['access_mode'] ?? 'open' );
            $output['paywall_limit_sec'] = absint( $input['paywall_limit_sec'] ?? 60 );
            $output['woocommerce_product_id'] = absint( $input['woocommerce_product_id'] ?? 0 );
            $output['vast_pre_roll'] = esc_url_raw( $input['vast_pre_roll'] ?? '' );
            $output['vast_mid_roll'] = esc_url_raw( $input['vast_mid_roll'] ?? '' );
            $output['vast_mid_roll_interval'] = absint( $input['vast_mid_roll_interval'] ?? 10 );
        }

        return $output;
    }

    public static function render_settings_page() {
        $active_tab = isset( $_GET['tab'] ) ? $_GET['tab'] : 'general';
        ?>
        <div class="wrap ucs-admin-wrap">
            <h1>Ultimate Content Streamer Settings</h1>
            
            <h2 class="nav-tab-wrapper">
                <a href="?page=ucs-settings&tab=general" class="nav-tab <?php echo $active_tab == 'general' ? 'nav-tab-active' : ''; ?>">General</a>
                <a href="?page=ucs-settings&tab=ads" class="nav-tab <?php echo $active_tab == 'ads' ? 'nav-tab-active' : ''; ?>">Advertising</a>
                <a href="?page=ucs-settings&tab=data" class="nav-tab <?php echo $active_tab == 'data' ? 'nav-tab-active' : ''; ?>">Data Import</a>
                <a href="?page=ucs-settings&tab=license" class="nav-tab <?php echo $active_tab == 'license' ? 'nav-tab-active' : ''; ?>">License Key</a>
            </h2>

            <?php 
            if ($active_tab == 'license') settings_errors( 'ucs_license_messages' );
            else settings_errors(); 
            ?>

            <form method="post" action="<?php echo ($active_tab === 'general' || $active_tab === 'ads') ? 'options.php' : ''; ?>">
                <?php
                if ( $active_tab === 'general' ) {
                    settings_fields( self::SETTINGS_GROUP );
                    do_settings_sections( self::SETTINGS_PAGE_SLUG );
                    // Don't show submit button if everything on this tab is locked (optional, but here we have mixed fields)
                    submit_button();
                } elseif ( $active_tab === 'ads' ) {
                    settings_fields( self::SETTINGS_GROUP );
                    do_settings_sections( 'ucs-ads-settings' );
                    submit_button();
                } elseif ( $active_tab === 'data' ) {
                    do_settings_sections( 'ucs-data-settings' );
                    self::render_csv_form(); 
                } else {
                    do_settings_sections( 'ucs-license-settings' );
                }
                ?>
            </form>
            <div id="ucs-import-status" class="notice ucs-admin-notice" style="display:none; margin-top:10px;"></div>
        </div>
        <?php
    }

    // --- LOCK HELPER ---
    private static function check_lock() {
        if ( ! UCS_License::is_valid() ) {
            echo ' disabled style="background-color:#eee; cursor:not-allowed;" ';
            return true;
        }
        return false;
    }

    // --- RENDERERS ---

    public static function render_access_mode_field() {
        $opts = get_option('ucs_settings'); $val = $opts['access_mode'] ?? 'open';
        echo '<select name="ucs_settings[access_mode]"'; self::check_lock(); echo '>';
        echo '<option value="open" '.selected($val,'open',false).'>Open</option>';
        echo '<option value="paid_hard" '.selected($val,'paid_hard',false).'>Paid Hard Gating</option>';
        echo '</select>';
        if(!UCS_License::is_valid()) echo ' <em style="color:#d63638;">(Activate License to Unlock)</em>';
    }

    public static function render_paywall_limit_field() {
        $opts = get_option('ucs_settings'); 
        echo '<input type="number" name="ucs_settings[paywall_limit_sec]" value="'.esc_attr($opts['paywall_limit_sec'] ?? 60).'"'; self::check_lock(); echo '>';
        if(!UCS_License::is_valid()) echo ' <em style="color:#d63638;">(Locked)</em>';
    }

    public static function render_wc_product_field() {
        $opts = get_option('ucs_settings'); 
        echo '<input type="number" name="ucs_settings[woocommerce_product_id]" value="'.esc_attr($opts['woocommerce_product_id'] ?? 0).'"'; self::check_lock(); echo '>';
        echo '<p class="description">Enter Product ID</p>';
    }

    public static function section_ads_cb() {
        if(!UCS_License::is_valid()) echo '<div class="notice notice-error inline"><p><strong>FEATURE LOCKED:</strong> Advertising settings are available for Premium users only.</p></div>';
        else echo '<p>Configure your Video Ad Serving Template (VAST) URLs here.</p>';
    }

    public static function render_ad_field($args) {
        $opts = get_option('ucs_settings'); 
        echo '<input type="url" class="regular-text" name="ucs_settings['.$args['key'].']" value="'.esc_attr($opts[$args['key']] ?? '').'"'; self::check_lock(); echo '>';
    }

    public static function render_ad_interval_field($args) {
        $opts = get_option('ucs_settings'); 
        echo '<input type="number" class="small-text" name="ucs_settings['.$args['key'].']" value="'.esc_attr($opts[$args['key']] ?? 10).'"'; self::check_lock(); echo '>';
    }

    // Free Field
    public static function render_embed_toggle_field() { 
        $options = get_option( 'ucs_settings' ); 
        $checked = isset( $options['embed_code_toggle'] ) && $options['embed_code_toggle'] === '1' ? 'checked' : ''; 
        echo '<label><input type="checkbox" name="ucs_settings[embed_code_toggle]" value="1" ' . $checked . '> Show "Get Embed Code" button</label>'; 
    }

    // License & Import Sections
    public static function section_license_cb() { echo '<p>Activate your license to unlock Data Import, Ads, and Monetization.</p>'; }
    
    public static function render_license_field() {
        $key = UCS_License::get_key();
        $valid = UCS_License::is_valid();
        wp_nonce_field( 'ucs_activate_license', 'ucs_license_nonce' );

        if ( $valid ) {
            echo '<div style="color:green; font-weight:bold; margin-bottom:10px;">ACTIVE</div>';
            echo '<input type="password" value="' . esc_attr( $key ) . '" class="regular-text" disabled> ';
            echo '<input type="submit" name="ucs_deactivate_btn" class="button" value="Deactivate">';
        } else {
            echo '<input type="text" name="ucs_license_key" class="regular-text" placeholder="Enter License Key"> ';
            echo '<input type="submit" name="ucs_activate_btn" class="button button-primary" value="Activate">';
            echo '<p class="description">Product ID: ' . UCS_PREMIUM_PRODUCT_ID . '</p>';
        }
    }

    public static function render_import_tools() {
        if ( ! UCS_License::is_valid() ) {
            echo '<p style="color:red;"><strong>Feature Locked:</strong> Please activate your license to use the Auto-Importer.</p>';
            echo '<button type="button" class="button" disabled>Run M3U Import (Locked)</button>';
        } else {
            echo '<button type="button" class="button" id="ucs-m3u-import-channels">Run M3U Import (Auto)</button>';
        }
    }

    public static function render_csv_form() {
        echo '<hr><h2>CSV Channel Upload</h2>';
        if ( ! UCS_License::is_valid() ) {
            echo '<p style="color:red;"><strong>Feature Locked:</strong> Please activate your license to use CSV Upload.</p>';
            return;
        }
        ?>
        <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" enctype="multipart/form-data">
            <input type="hidden" name="action" value="ucs_handle_csv_upload">
            <?php wp_nonce_field( 'ucs_csv_upload_nonce', 'ucs_csv_nonce' ); ?>
            <input type="file" name="ucs_csv_file" accept=".csv" required>
            <?php submit_button( 'Upload CSV', 'secondary', 'submit', false ); ?>
        </form>
        <?php
    }

    public static function render_analytics_page() { /* Analytics code from previous response... */ echo '<div class="wrap"><h1>Analytics</h1><p>Active License required for real-time data.</p></div>'; }
}