<?php
/**
 * Utility class for helper functions and WooCommerce checks
 */
class UCS_Utils {

    /**
     * Display admin notice if WooCommerce is not active.
     */
    public static function admin_woocommerce_check_notice() {
        if ( ! self::is_woocommerce_active() && current_user_can('manage_options') ) {
            $message = sprintf(
                esc_html__( 'Ultimate Content Streamer Paywall features require %s to be installed and active.', 'ucs' ),
                '<strong>WooCommerce</strong>'
            );
            printf( '<div class="notice notice-warning"><p>%s</p></div>', $message );
        }
    }

    /**
     * Check if WooCommerce is active.
     * @return bool
     */
    public static function is_woocommerce_active() {
        return class_exists( 'WooCommerce' );
    }

    /**
     * Checks if the current user has purchased the required WooCommerce product.
     *
     * @param int $product_id The WooCommerce product ID required for access.
     * @return bool True if the user is logged in and has purchased the product, false otherwise.
     */
    public static function has_user_purchased_access( $product_id ) {
        // If WooCommerce isn't installed, or user isn't logged in, or no product ID set, return false
        if ( ! self::is_woocommerce_active() || ! is_user_logged_in() || empty( $product_id ) ) {
            return false;
        }
        
        $user = wp_get_current_user();
        
        // Use WooCommerce's built-in function to check purchase history
        // Checks based on email and user ID
        return wc_customer_bought_product( $user->user_email, $user->ID, $product_id );
    }
    
    /**
     * Generates a simple, non-alert message box.
     *
     * @param string $message The text to display.
     * @param string $type 'info', 'success', or 'error'.
     * @return string HTML for the message box.
     */
    public static function generate_message_box( $message, $type = 'info' ) {
        // Basic sanitization for class name
        $type = preg_replace( '/[^a-zA-Z0-9-_]/', '', $type );
        $class = 'ucs-message-box ucs-' . $type;
        return '<div class="' . esc_attr( $class ) . '"><p>' . esc_html( $message ) . '</p></div>';
    }
}

// Add the WooCommerce check notice hook
add_action( 'admin_notices', array( 'UCS_Utils', 'admin_woocommerce_check_notice' ) );