/**
 * JavaScript for the Frontend Chatbot
 */
document.addEventListener('DOMContentLoaded', function() {
    
    // ucmr_chatbot_ajax is localized from PHP
    if (typeof ucmr_chatbot_ajax === 'undefined') {
        return; // Don't run if data is missing
    }

    // --- 1. Create Chatbot HTML Elements ---
    
    // Toggle Button
    const toggleButton = document.createElement('button');
    toggleButton.id = 'ucmr-chat-toggle';
    toggleButton.innerHTML = '&#9998;'; // Pencil icon
    document.body.appendChild(toggleButton);

    // Chat Widget
    const widget = document.createElement('div');
    widget.id = 'ucmr-chat-widget';
    
    widget.innerHTML = `
        <div id="ucmr-chat-header">Ask about this article</div>
        <div id="ucmr-chat-messages">
            <div class="ucmr-chat-bubble-fe ucmr-chat-ai-fe">
                Hi! Ask me any questions you have about this article.
            </div>
        </div>
        <div id="ucmr-chat-input-area">
            <input type="text" id="ucmr-chat-input" placeholder="Type your question...">
            <button id="ucmr-chat-send">&rarr;</button>
        </div>
    `;
    document.body.appendChild(widget);

    // --- 2. Get Element References ---
    const messagesBox = document.getElementById('ucmr-chat-messages');
    const inputField = document.getElementById('ucmr-chat-input');
    const sendButton = document.getElementById('ucmr-chat-send');
    const chatHeader = document.getElementById('ucmr-chat-header');

    // --- 3. Add Event Listeners ---
    
    // Toggle widget visibility
    toggleButton.addEventListener('click', () => {
        widget.classList.toggle('ucmr-chat-open');
    });

    // Send message on button click
    sendButton.addEventListener('click', sendMessage);

    // Send message on Enter key
    inputField.addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            sendMessage();
        }
    });

    // --- 4. Helper Functions ---
    
    function addMessage(message, sender) {
        const bubble = document.createElement('div');
        bubble.classList.add('ucmr-chat-bubble-fe');
        bubble.classList.add(sender === 'user' ? 'ucmr-chat-user-fe' : 'ucmr-chat-ai-fe');
        bubble.textContent = message;
        messagesBox.appendChild(bubble);
        
        // Scroll to bottom
        messagesBox.scrollTop = messagesBox.scrollHeight;
    }

    function sendMessage() {
        const question = inputField.value.trim();
        if (question === '') {
            return;
        }

        // Add user message to UI
        addMessage(question, 'user');
        inputField.value = '';
        sendButton.disabled = true;

        // Add loading bubble
        const loadingBubble = document.createElement('div');
        loadingBubble.classList.add('ucmr-chat-bubble-fe', 'ucmr-chat-ai-fe');
        loadingBubble.textContent = '...';
        messagesBox.appendChild(loadingBubble);
        messagesBox.scrollTop = messagesBox.scrollHeight;

        // Prepare data for AJAX
        const formData = new URLSearchParams();
        formData.append('action', 'ucmr_chatbot_query');
        formData.append('_ajax_nonce', ucmr_chatbot_ajax.nonce);
        formData.append('post_id', ucmr_chatbot_ajax.post_id);
        formData.append('question', question);

        // Send AJAX request
        fetch(ucmr_chatbot_ajax.ajax_url, {
            method: 'POST',
            body: formData,
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                loadingBubble.textContent = data.data.reply;
            } else {
                loadingBubble.textContent = 'Error: ' + data.data.message;
            }
            sendButton.disabled = false;
        })
        .catch(error => {
            loadingBubble.textContent = 'Error: Request failed. Please try again.';
            sendButton.disabled = false;
        });
    }

});