<?php
if ( ! defined( 'WPINC' ) ) { die; }

class UCMR_AI_Client {
    private $options;

    public function __construct( $options ) {
        $this->options = $options;
    }

    /**
     * Main entry point with FALLBACK logic.
     */
    public function generate_content( $user_prompt, $system_prompt = '', $provider_override = null ) {
        
        $provider = $provider_override ? $provider_override : ( $this->options['ai_provider'] ?? 'gemini' );
        $model = $this->options['ai_model'] ?? ''; // Allow specific model override
        $temperature = (float) ( $this->options['ai_temperature'] ?? 0.7 );

        // Try primary provider
        $result = $this->call_provider( $provider, $user_prompt, $system_prompt, $model, $temperature );

        if ( ! is_wp_error( $result ) ) {
            return $result;
        }

        // --- FALLBACK LOGIC ---
        // If primary fails, check other keys and try them in order of preference.
        $fallbacks = ['gemini', 'chatgpt', 'claude', 'grok', 'deepseek'];
        
        foreach ( $fallbacks as $fallback_provider ) {
            if ( $fallback_provider === $provider ) continue; // Skip failed primary
            
            // Check if we have a key for this fallback
            $key_name = $fallback_provider === 'chatgpt' ? 'chatgpt_api_key' : $fallback_provider . '_api_key';
            if ( ! empty( $this->options[$key_name] ) ) {
                error_log( "[UCMR] Primary AI ($provider) failed. Falling back to $fallback_provider." );
                $result = $this->call_provider( $fallback_provider, $user_prompt, $system_prompt, '', $temperature );
                if ( ! is_wp_error( $result ) ) {
                    return $result;
                }
            }
        }

        return $result; // Return the last error if all failed
    }

    private function call_provider( $provider, $user, $system, $model, $temp ) {
        switch ( $provider ) {
            case 'gemini': 
                // Updated default to a stable versioned ID to prevent 404 errors
                return $this->call_gemini( $user, $system, $model ?: 'gemini-1.5-flash-001', $temp );
            case 'chatgpt': 
                return $this->call_openai( $user, $system, $model ?: 'gpt-4o-mini', $temp );
            case 'claude': 
                return $this->call_claude( $user, $system, $model ?: 'claude-3-haiku-20240307', $temp );
            case 'grok':
                return $this->call_openai_compatible( $user, $system, 'grok-beta', $temp, 'https://api.x.ai/v1/chat/completions', $this->options['grok_api_key'] );
            case 'deepseek':
                return $this->call_openai_compatible( $user, $system, 'deepseek-chat', $temp, 'https://api.deepseek.com/chat/completions', $this->options['deepseek_api_key'] );
            case 'meta': 
                 return new WP_Error( 'not_implemented', 'Meta provider not configured.' );
            default: 
                return new WP_Error( 'invalid_provider', 'Invalid AI provider.' );
        }
    }

    private function call_openai_compatible( $user, $system, $model, $temp, $url, $key ) {
        if ( empty( $key ) ) return new WP_Error( 'missing_key', 'API key missing.' );
        
        $messages = [];
        if ( $system ) $messages[] = [ 'role' => 'system', 'content' => $system ];
        $messages[] = [ 'role' => 'user', 'content' => $user ];

        $response = wp_remote_post( $url, [
            'headers' => [ 'Authorization' => 'Bearer ' . $key, 'Content-Type' => 'application/json' ],
            'body' => json_encode( [ 'model' => $model, 'messages' => $messages, 'temperature' => $temp ] ),
            'timeout' => 45
        ]);

        return $this->parse_response( $response, 'openai' );
    }

    private function call_gemini( $user_prompt, $system_prompt, $model, $temperature ) {
        $api_key = $this->options['gemini_api_key'] ?? '';
        if ( empty( $api_key ) ) return new WP_Error( 'missing_key', 'Gemini API key is missing.' );
        
        $api_url = 'https://generativelanguage.googleapis.com/v1beta/models/' . $model . ':generateContent?key=' . $api_key;
        
        $body = [
            'contents' => [[ 'role' => 'user', 'parts' => [[ 'text' => $user_prompt ]] ]],
            'generationConfig' => [ 'temperature' => $temperature ]
        ];
        if ( ! empty( $system_prompt ) ) $body['systemInstruction'] = [ 'parts' => [[ 'text' => $system_prompt ]] ];
        
        $response = wp_remote_post( $api_url, [ 'headers' => [ 'Content-Type' => 'application/json' ], 'body' => json_encode( $body ), 'timeout' => 45 ] );
        return $this->parse_response( $response, 'gemini' );
    }
    
    private function call_openai( $user_prompt, $system_prompt, $model, $temperature ) {
        $api_key = $this->options['chatgpt_api_key'] ?? '';
        if ( empty( $api_key ) ) return new WP_Error( 'missing_key', 'OpenAI API key is missing.' );
        $api_url = 'https://api.openai.com/v1/chat/completions';
        $messages = [];
        if ( ! empty( $system_prompt ) ) $messages[] = [ 'role' => 'system', 'content' => $system_prompt ];
        $messages[] = [ 'role' => 'user', 'content' => $user_prompt ];
        $body = [ 'model' => $model, 'messages' => $messages, 'temperature' => $temperature ];
        
        $response = wp_remote_post( $api_url, [ 'headers' => [ 'Content-Type' => 'application/json', 'Authorization' => 'Bearer ' . $api_key ], 'body' => json_encode( $body ), 'timeout' => 45 ] );
        return $this->parse_response( $response, 'openai' );
    }

    private function call_claude( $user_prompt, $system_prompt, $model, $temperature ) {
        $api_key = $this->options['claude_api_key'] ?? '';
        if ( empty( $api_key ) ) return new WP_Error( 'missing_key', 'Claude API key is missing.' );
        $api_url = 'https://api.anthropic.com/v1/messages';
        $body = [ 'model' => $model, 'messages' => [[ 'role' => 'user', 'content' => $user_prompt ]], 'temperature' => $temperature, 'max_tokens' => 4096 ];
        if ( ! empty( $system_prompt ) ) $body['system'] = $system_prompt;
        
        $response = wp_remote_post( $api_url, [ 'headers' => [ 'Content-Type' => 'application/json', 'x-api-key' => $api_key, 'anthropic-version' => '2023-06-01' ], 'body' => json_encode( $body ), 'timeout' => 45 ] );
        return $this->parse_response( $response, 'claude' );
    }

    public function generate_image_content( $prompt, $model = 'dall-e-3' ) {
        // Simple DALL-E integration
        $api_key = $this->options['chatgpt_api_key'] ?? '';
        if ( empty( $api_key ) ) return new WP_Error( 'missing_key', 'OpenAI API key needed for DALL-E.' );
        $response = wp_remote_post( 'https://api.openai.com/v1/images/generations', [
            'headers' => [ 'Authorization' => 'Bearer ' . $api_key, 'Content-Type' => 'application/json' ],
            'body' => json_encode([ 'model' => 'dall-e-3', 'prompt' => $prompt, 'n' => 1, 'size' => '1024x1024' ]),
            'timeout' => 60
        ]);
        if ( is_wp_error( $response ) ) return $response;
        $data = json_decode( wp_remote_retrieve_body( $response ) );
        return $data->data[0]->url ?? new WP_Error('api_error', 'Image gen failed');
    }
    
    // NEW: Nano-Banana Image Editor (gemini-2.5-flash-image-preview placeholder logic)
    public function generate_image_variation( $prompt, $base64_image, $mime_type ) {
        // This functionality uses Gemini's vision capabilities.
        // Currently relying on standard generateContent with image input.
        $api_key = $this->options['gemini_api_key'] ?? '';
        if ( empty( $api_key ) ) return new WP_Error( 'missing_key', 'Gemini API key missing.' );
        
        $model = 'gemini-1.5-flash-latest'; // Use a vision-capable model
        $api_url = 'https://generativelanguage.googleapis.com/v1beta/models/' . $model . ':generateContent?key=' . $api_key;
        
        $body = [
            'contents' => [[
                'parts' => [
                    [ 'text' => "Edit this image: " . $prompt ],
                    [ 'inlineData' => [ 'mimeType' => $mime_type, 'data' => $base64_image ] ]
                ]
            ]]
        ];

        $response = wp_remote_post( $api_url, [
            'headers' => [ 'Content-Type' => 'application/json' ],
            'body' => json_encode( $body ),
            'timeout' => 60
        ]);
        
        // Note: Gemini text models return text about images, not edited images directly in this endpoint.
        // This is a placeholder until Google releases a direct "edit image" endpoint for Gemini.
        return new WP_Error('not_supported', 'Direct image editing API not yet available for this model.');
    }

    private function parse_response( $response, $provider ) {
        if ( is_wp_error( $response ) ) return $response;
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body );
        
        // Debugging: Log full response on error
        if ( isset( $data->error ) ) {
            return new WP_Error( 'api_error', print_r( $data->error, true ) );
        }
        
        if($provider === 'gemini') return $data->candidates[0]->content->parts[0]->text ?? new WP_Error('api_error', 'Invalid Gemini response: ' . print_r($data, true));
        if($provider === 'openai') return $data->choices[0]->message->content ?? new WP_Error('api_error', 'Invalid OpenAI response: ' . print_r($data, true));
        if($provider === 'claude') return $data->content[0]->text ?? new WP_Error('api_error', 'Invalid Claude response: ' . print_r($data, true));
        
        return new WP_Error('unknown', 'Unknown provider');
    }
}