<?php
/**
 * UCMR_Audio_Client Class
 *
 * Handles Text-to-Speech (TTS) generation using OpenAI's Audio API (tts-1-hd)
 * and integrates with ElevenLabs for advanced voice cloning.
 *
 * It provides methods for generating standard high-quality audio and
 * for initiating and using cloned voices.
 *
 * @package UltimateContentMoneyRiver
 * @subpackage Includes
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
    die;
}

// We need these core WP files to handle audio sideloading
require_once( ABSPATH . 'wp-admin/includes/file.php' );
require_once( ABSPATH . 'wp-admin/includes/media.php' );
require_once( ABSPATH . 'wp-admin/includes/image.php' ); // for media_handle_sideload

class UCMR_Audio_Client {

    private $openai_api_key;
    private $elevenlabs_api_key;
    private $elevenlabs_voice_id; // For specific voice cloning

    public function __construct() {
        $options = get_option( 'ucmr_options' );
        $this->openai_api_key   = $options['chatgpt_api_key'] ?? ''; // OpenAI key is used for TTS
        $this->elevenlabs_api_key = $options['elevenlabs_api_key'] ?? '';
        
        // Find the selected cloned voice ID
        $auto_voice = $options['style_auto_audio_voice'] ?? '';
        if( strpos($auto_voice, 'elevenlabs:cloned:') === 0 ) {
            $this->elevenlabs_voice_id = str_replace('elevenlabs:cloned:', '', $auto_voice);
        } else {
            // Check for the "My Cloned Voice" default, which uses a saved option
            $cloned_voice_option = get_option('ucmr_elevenlabs_cloned_voice_id'); // This is set during cloning
            if ($auto_voice === 'elevenlabs:cloned' && $cloned_voice_option) {
                 $this->elevenlabs_voice_id = $cloned_voice_option;
            }
        }
    }

    /**
     * Generates speech from text using OpenAI's TTS API.
     *
     * @param string $text The text to convert to speech.
     * @param string $voice The OpenAI voice to use (alloy, echo, fable, onyx, nova, shimmer).
     * @param string $model The OpenAI TTS model (tts-1 or tts-1-hd).
     * @return string|WP_Error Raw audio data on success, WP_Error on failure.
     */
    public function generate_speech( $text, $voice = 'nova', $model = 'tts-1-hd' ) {
        if ( empty( $this->openai_api_key ) ) {
            return new WP_Error( 'no_openai_key', 'OpenAI API key is not set for TTS.' );
        }

        $url = 'https://api.openai.com/v1/audio/speech';
        $headers = [
            'Authorization' => 'Bearer ' . $this->openai_api_key,
            'Content-Type'  => 'application/json',
        ];
        $body = [
            'model' => $model,
            'input' => $text,
            'voice' => $voice,
            'response_format' => 'mp3'
        ];

        $response = wp_remote_post( $url, [
            'headers' => $headers,
            'body'    => json_encode( $body ),
            'timeout' => 60, // Increased timeout for audio generation
        ] );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );

        if ( $response_code === 200 ) {
            // Return raw MP3 data
            return $response_body;
        } else {
            $error_data = json_decode( $response_body, true );
            return new WP_Error( 'openai_tts_error', isset( $error_data['error']['message'] ) ? $error_data['error']['message'] : 'Unknown OpenAI TTS error.', [ 'status' => $response_code ] );
        }
    }

    /**
     * Creates an Instant Voice Clone using ElevenLabs.
     *
     * @param string $voice_name A name for the new voice.
     * @param string $audio_base64 Base64 encoded audio file content.
     * @param string $mime_type The MIME type of the audio file (e.g., 'audio/mpeg').
     * @return string|WP_Error The new voice_id on success, WP_Error on failure.
     */
    public function clone_voice( $voice_name, $audio_base64, $mime_type ) {
        if ( empty( $this->elevenlabs_api_key ) ) {
            return new WP_Error( 'no_elevenlabs_key', 'ElevenLabs API key is not set for voice cloning.' );
        }

        $url = 'https://api.elevenlabs.io/v1/voices/add';
        
        // Decode base64 audio to binary string
        $audio_data = base64_decode( $audio_base64 );
        if ( $audio_data === false ) {
            return new WP_Error( 'invalid_audio_data', 'Failed to decode base64 audio data.' );
        }

        // Create a temporary file to send to ElevenLabs
        $tmp_file = wp_tempnam( 'ucmr_clone' );
        if ( ! $tmp_file ) {
            return new WP_Error( 'temp_file_error', 'Could not create temporary file for audio upload.' );
        }
        
        file_put_contents( $tmp_file, $audio_data );
        
        $filename = basename( $tmp_file ) . '.mp3'; // Give it a proper extension
        $boundary = '----' . md5( time() );

        $body = '';
        // name
        $body .= '--' . $boundary . "\r\n";
        $body .= 'Content-Disposition: form-data; name="name"' . "\r\n\r\n";
        $body .= $voice_name . "\r\n";
        
        // files
        $body .= '--' . $boundary . "\r\n";
        $body .= 'Content-Disposition: form-data; name="files"; filename="' . $filename . '"' . "\r\n";
        $body .= 'Content-Type: ' . $mime_type . "\r\n\r\n";
        $body .= $audio_data . "\r\n"; // Add the raw file data
        
        // end
        $body .= '--' . $boundary . '--' . "\r\n";

        $response = wp_remote_post( $url, [
            'headers' => [
                'xi-api-key'   => $this->elevenlabs_api_key,
                'Content-Type' => 'multipart/form-data; boundary=' . $boundary,
            ],
            'body'    => $body,
            'timeout' => 90,
        ] );

        unlink( $tmp_file ); // Clean up temp file

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );
        $data = json_decode( $response_body, true );

        if ( $response_code === 200 && isset( $data['voice_id'] ) ) {
            // Save this new voice ID as the "My Cloned Voice" default
            update_option( 'ucmr_elevenlabs_cloned_voice_id', $data['voice_id'] );
            return $data['voice_id'];
        } else {
            return new WP_Error( 'elevenlabs_clone_error', isset( $data['detail']['message'] ) ? $data['detail']['message'] : 'Unknown ElevenLabs voice cloning error.', [ 'status' => $response_code, 'response_data' => $data ] );
        }
    }

    /**
     * Generates speech using a cloned ElevenLabs voice.
     *
     * @param string $text The text to convert to speech.
     * @param string $voice_id The ElevenLabs voice ID to use.
     * @return string|WP_Error Raw audio data on success, WP_Error on failure.
     */
    public function generate_cloned_speech( $text, $voice_id = null ) {
        if ( empty( $this->elevenlabs_api_key ) ) {
            return new WP_Error( 'no_elevenlabs_key', 'ElevenLabs API key is not set for cloned voice TTS.' );
        }
        
        $voice_id_to_use = $voice_id ? $voice_id : $this->elevenlabs_voice_id;
        
        if ( empty( $voice_id_to_use ) ) {
            return new WP_Error( 'no_cloned_voice_id', 'No ElevenLabs cloned voice ID available or selected.' );
        }

        $url = "https://api.elevenlabs.io/v1/text-to-speech/{$voice_id_to_use}";
        $headers = [
            'xi-api-key' => $this->elevenlabs_api_key,
            'Content-Type'  => 'application/json',
            'Accept'        => 'audio/mpeg', // Request MP3 directly
        ];
        $body = [
            'text' => $text,
            'model_id' => 'eleven_multilingual_v2', // Or 'eleven_monolingual_v1'
            'voice_settings' => [
                'stability' => 0.5,
                'similarity_boost' => 0.75,
            ],
        ];

        $response = wp_remote_post( $url, [
            'headers' => $headers,
            'body'    => json_encode( $body ),
            'timeout' => 60,
        ] );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );

        if ( $response_code === 200 ) {
            // Return raw MP3 data
            return $response_body;
        } else {
            $error_data = json_decode( $response_body, true );
            return new WP_Error( 'elevenlabs_tts_error', isset( $error_data['detail']['message'] ) ? $error_data['detail']['message'] : 'Unknown ElevenLabs TTS error.', [ 'status' => $response_code ] );
        }
    }

    /**
     * Get available ElevenLabs voices (including cloned ones).
     * @return array|WP_Error
     */
    public function get_elevenlabs_voices() {
        if ( empty( $this->elevenlabs_api_key ) ) {
            return new WP_Error( 'no_elevenlabs_key', 'ElevenLabs API key is not set.' );
        }

        $url = 'https://api.elevenlabs.io/v1/voices';
        $headers = [
            'xi-api-key' => $this->elevenlabs_api_key,
        ];

        $response = wp_remote_get( $url, [
            'headers' => $headers,
            'timeout' => 30,
        ] );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );
        $data = json_decode( $response_body, true );

        if ( $response_code === 200 && isset( $data['voices'] ) ) {
            $voices = [];
            foreach ( $data['voices'] as $voice ) {
                $voices[] = [
                    'voice_id' => $voice['voice_id'],
                    'name' => $voice['name'],
                    'category' => $voice['category'], // pre-made, cloned, generated
                ];
            }
            return $voices;
        } else {
            return new WP_Error( 'elevenlabs_voices_error', isset( $data['detail']['message'] ) ? $data['detail']['message'] : 'Unknown ElevenLabs voice fetch error.', [ 'status' => $response_code ] );
        }
    }
    
    /**
     * Uploads raw audio data to the WordPress Media Library.
     *
     * @param string $audio_data Raw MP3 audio data.
     * @param string $filename The desired filename (e.g., 'post-audio-123.mp3').
     * @param int $post_id The post ID to attach the audio to.
     * @param string $title The title for the media item.
     *
     * @return int|WP_Error The attachment ID on success, or WP_Error on failure.
     */
    public function upload_audio_to_media_library( $audio_data, $filename, $post_id, $title ) {
        $upload_dir = wp_upload_dir();
        $upload_path = $upload_dir['path'] . '/' . $filename;

        // Save the raw audio data to a temporary file
        $bytes_written = file_put_contents( $upload_path, $audio_data );
        if ( $bytes_written === false ) {
            return new WP_Error( 'file_write_error', 'Could not write audio data to temporary file.' );
        }

        $file_type = wp_check_filetype( $filename, null );

        $attachment_data = [
            'post_mime_type' => $file_type['type'],
            'post_title'     => $title,
            'post_content'   => '',
            'post_status'    => 'inherit',
        ];

        // Insert the attachment
        $attachment_id = wp_insert_attachment( $attachment_data, $upload_path, $post_id );

        if ( is_wp_error( $attachment_id ) ) {
            unlink( $upload_path ); // Clean up
            return $attachment_id;
        }

        // Generate attachment metadata
        $attachment_meta = wp_generate_attachment_metadata( $attachment_id, $upload_path );
        wp_update_attachment_metadata( $attachment_id, $attachment_meta );

        return $attachment_id;
    }
}