<?php
if ( ! defined( 'WPINC' ) ) { die; }

class UCMR_Content_Generator {
    private $options;
    private $ai_client;
    private $media_client;
    private $audio_client;

    public function __construct( $options, $ai_client, $media_client, $audio_client ) {
        $this->options      = $options;
        $this->ai_client    = $ai_client;
        $this->media_client = $media_client;
        $this->audio_client = $audio_client;
    }

    /**
     * Automated generation for Scheduler.
     */
    public function generate_post_from_keywords_automated( $keywords, $settings ) {
        // 1. Build prompt with length/tone constraints
        $length_prompt = "Length: Approx " . ($settings['length'] ?? 1500) . " words.";
        $tone = $settings['tone'] ?? 'professional';
        $sys_prompt = "You are an expert content writer. Write in a {$tone} tone. {$length_prompt}. Format using HTML (h2, h3, p).";
        $user_prompt = "Write a comprehensive article about: " . $keywords;
        
        // 2. Generate Text
        $ai_result = $this->ai_client->generate_content( $user_prompt, $sys_prompt );
        if ( is_wp_error( $ai_result ) ) return; // Log error

        // 3. Parse Title
        $title_pattern = '/\\[Title:\s*(.*?)\\]/i';
        preg_match($title_pattern, $ai_result, $matches);
        $title = $matches[1] ?? 'Auto: ' . $keywords;
        $content = trim(preg_replace($title_pattern, '', $ai_result));

        // 4. "Publish Day Before" Logic (Schedule for +24 hours)
        $publish_date = date('Y-m-d H:i:s', strtotime('+1 day'));

        // 5. Insert Post
        $post_data = [
            'post_title'    => $title,
            'post_content'  => $content,
            'post_status'   => 'future', // Scheduled
            'post_date'     => $publish_date,
            'post_author'   => $settings['author'] ?? 1,
            'post_category' => [ $settings['category'] ?? 1 ]
        ];
        $pid = wp_insert_post($post_data);
        
        // 6. Media Handling (Balanced)
        $img_source = $settings['img_source'] ?? 'hybrid';
        $img_count = $settings['img_count'] ?? 2;
        
        if ( $img_source === 'ai' ) {
            // Force AI generation
            $this->generate_ai_images_for_post($pid, $keywords, $img_count);
        } else {
            // Try Royalty Free first
            $found = $this->media_client->attach_images_to_post($pid, $keywords, $img_count);
            // If Hybrid and no images found, fallback to AI
            if ( $img_source === 'hybrid' && $found === 0 ) {
                $this->generate_ai_images_for_post($pid, $keywords, $img_count);
            }
        }

        // 7. Podcast
        if(!empty($settings['podcast'])) {
            $this->generate_and_attach_audio($pid, $content);
        }
    }

    private function generate_ai_images_for_post($post_id, $prompt, $count) {
        // Simple loop to generate N images
        for($i=0; $i<$count; $i++) {
            $img_url = $this->ai_client->generate_image_content("A high quality editorial image for an article about: " . $prompt);
            if(!is_wp_error($img_url)) {
                // Sideload this URL
                $this->media_client->sideload_image($img_url, $post_id, $prompt);
                if($i===0) set_post_thumbnail($post_id, $this->media_client->get_last_attachment_id());
            }
        }
    }
    
    // ... (Include save_generated_post, generate_post_from_keywords, generate_and_attach_audio from previous versions) ...
    public function save_generated_post( $title, $content, $status, $cat_id, $new_cat, $keywords ) {
        // ... (Same as previous step) ...
        // Re-included for completeness
        $category_ids = [];
        if ( $new_cat ) {
            $term = wp_insert_term( $new_cat, 'category' );
            if ( ! is_wp_error( $term ) ) $category_ids[] = $term['term_id'];
        } elseif ( $cat_id ) {
            $category_ids[] = $cat_id;
        }

        $post_data = [
            'post_title'    => sanitize_text_field( $title ),
            'post_content'  => wp_kses_post( $content ),
            'post_status'   => $status,
            'post_author'   => get_current_user_id(),
            'post_type'     => 'post',
            'post_category' => $category_ids,
        ];
        $post_id = wp_insert_post( $post_data, true );

        if ( is_wp_error( $post_id ) ) return $post_id;

        $this->media_client->attach_images_to_post( $post_id, $keywords ); // Default media logic for single post
        
        // Auto-Audio check
        if ( ( $this->options['style_auto_audio'] ?? '0' ) === '1' ) {
            $this->generate_and_attach_audio( $post_id, $content );
        }

        return [ 'post_id' => $post_id, 'audio_url' => '' ];
    }

    private function generate_and_attach_audio( $post_id, $content ) {
        $text = wp_strip_all_tags( $content );
        $text = substr( $text, 0, 4000 );
        $audio_data = $this->audio_client->generate_speech( $text ); // Default voice
        if ( ! is_wp_error( $audio_data ) ) {
            $filename = 'audio-' . $post_id . '.mp3';
            $att_id = $this->audio_client->upload_audio_to_media_library( $audio_data, $filename, $post_id, 'Podcast' );
            if(!is_wp_error($att_id)) {
                $url = wp_get_attachment_url($att_id);
                update_post_meta( $post_id, '_ucmr_audio_url', $url );
                // Add to podcast taxonomy
                wp_set_object_terms( $post_id, 'Audio Articles', 'ucmr_podcast', true );
                return $url;
            }
        }
        return false;
    }
}