<?php
/**
 * UCMR Media Client
 *
 * This class handles all API communications with royalty-free image providers
 * like Pixabay, Pexels, and Unsplash.
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
    die;
}

// We need these core WP files to handle image sideloading
require_once( ABSPATH . 'wp-admin/includes/file.php' );
require_once( ABSPATH . 'wp-admin/includes/media.php' );
require_once( ABSPATH . 'wp-admin/includes/image.php' );

class UCMR_Media_Client {

    /**
     * The plugin options array.
     * @var array
     */
    private $options;

    /**
     * Constructor.
     *
     * @param array $options The plugin options array.
     */
    public function __construct( $options ) {
        $this->options = $options;
    }

    /**
     * Main entry point to find and attach images to a post.
     * It sets the first image found as the Featured Image.
     *
     * @param int $post_id The ID of the post to attach images to.
     * @param string $keywords The keywords to search for.
     */
    public function attach_images_to_post( $post_id, $keywords ) {
        $image_url = false;

        // Try Pixabay first
        if ( ! empty( $this->options['media_pixabay_key'] ) ) {
            $image_url = $this->get_pixabay_image( $keywords );
        }

        // Try Pexels if Pixabay failed
        if ( ! $image_url && ! empty( $this->options['media_pexels_key'] ) ) {
            $image_url = $this->get_pexels_image( $keywords );
        }

        // Try Unsplash if Pexels failed
        if ( ! $image_url && ! empty( $this->options['media_unsplash_key'] ) ) {
            $image_url = $this->get_unsplash_image( $keywords );
        }

        // If we found an image, download and set it as featured image
        if ( $image_url ) {
            $this->sideload_image_and_set_thumbnail( $image_url, $post_id, $keywords );
        }

        // TODO: The spec calls for 2-5 images.
        // This would require looping and appending HTML to the post content,
        // which is complex. For now, we set the Featured Image.
    }

    /**
     * Fetches an image from Pixabay.
     */
    private function get_pixabay_image( $keywords ) {
        $api_url = add_query_arg( [
            'key'        => $this->options['media_pixabay_key'],
            'q'          => urlencode( $keywords ),
            'image_type' => 'photo',
            'per_page'   => 5,
            'safesearch' => 'true'
        ], 'https://pixabay.com/api/' );

        $response = wp_remote_get( $api_url );
        if ( is_wp_error( $response ) ) return false;

        $data = json_decode( wp_remote_retrieve_body( $response ), true );
        if ( ! empty( $data['hits'][0]['largeImageURL'] ) ) {
            return $data['hits'][0]['largeImageURL'];
        }
        return false;
    }

    /**
     * Fetches an image from Pexels.
     */
    private function get_pexels_image( $keywords ) {
        $api_url = add_query_arg( [
            'query'    => urlencode( $keywords ),
            'per_page' => 1
        ], 'https://api.pexels.com/v1/search' );

        $response = wp_remote_get( $api_url, [
            'headers' => [ 'Authorization' => $this->options['media_pexels_key'] ]
        ] );

        if ( is_wp_error( $response ) ) return false;

        $data = json_decode( wp_remote_retrieve_body( $response ), true );
        if ( ! empty( $data['photos'][0]['src']['large'] ) ) {
            return $data['photos'][0]['src']['large'];
        }
        return false;
    }

    /**
     * Fetches an image from Unsplash.
     */
    private function get_unsplash_image( $keywords ) {
        $api_url = add_query_arg( [
            'client_id' => $this->options['media_unsplash_key'],
            'query'     => urlencode( $keywords ),
            'per_page'  => 1
        ], 'https://api.unsplash.com/search/photos' );

        $response = wp_remote_get( $api_url );
        if ( is_wp_error( $response ) ) return false;

        $data = json_decode( wp_remote_retrieve_body( $response ), true );
        if ( ! empty( $data['results'][0]['urls']['regular'] ) ) {
            return $data['results'][0]['urls']['regular'];
        }
        return false;
    }

    /**
     * Downloads an image from a URL, adds it to the Media Library,
     * and sets it as the post's featured image.
     *
     * @param string $url The URL of the image.
     * @param int $post_id The post ID.
     * @param string $keywords Description for alt text.
     */
    private function sideload_image_and_set_thumbnail( $url, $post_id, $keywords ) {
        // Sideload the image
        $attachment_id = media_handle_sideload( [ 'file' => $url, 'tmp_name' => download_url( $url ) ], $post_id, $keywords );

        if ( ! is_wp_error( $attachment_id ) ) {
            // Set it as the featured image
            set_post_thumbnail( $post_id, $attachment_id );
            
            // Add alt text
            update_post_meta( $attachment_id, '_wp_attachment_image_alt', $keywords );
        }
    }
}