<?php
/**
 * UCMR_Scheduler Class
 *
 * Handles the background processing of scheduled auto-posting jobs.
 * It runs via WP Cron, checks for due jobs, fetches fresh topics (from YouTube/RSS/Keywords),
 * injects affiliate links/custom prompts, and triggers the content generator.
 *
 * @package UltimateContentMoneyRiver
 */

if ( ! defined( 'WPINC' ) ) {
    die;
}

class UCMR_Scheduler {

    /**
     * @var UCMR_Content_Generator
     */
    private $content_gen;

    /**
     * Constructor.
     *
     * @param UCMR_Content_Generator $content_gen Dependency for generating the actual post.
     */
    public function __construct( $content_gen ) {
        $this->content_gen = $content_gen;
    }

    /**
     * Main Cron Callback.
     * Checks for active jobs due for execution and processes them.
     */
    public function process_jobs() {
        global $wpdb;
        $table = $wpdb->prefix . 'ucmr_scheduled_jobs';
        
        // Find active jobs where next_run is due (or past due)
        $jobs = $wpdb->get_results( "SELECT * FROM $table WHERE status = 'active' AND next_run <= NOW()" );

        if ( empty( $jobs ) ) {
            return;
        }

        $options = get_option( 'ucmr_options' );
        $youtube_key = $options['media_youtube_key'] ?? '';

        foreach ( $jobs as $job ) {
            $settings = json_decode( $job->settings, true );
            $history  = json_decode( $job->generated_history, true ) ?: [];
            
            $base_input = $job->topic_prompt;
            $topic_to_generate = '';
            $new_history_item = '';

            // --- 1. DETERMINE TOPIC BASED ON SOURCE TYPE ---

            // A. YouTube Channel Monitor
            if ( isset( $settings['source_type'] ) && $settings['source_type'] === 'youtube_channel' ) {
                if ( ! empty( $youtube_key ) ) {
                    $video_data = $this->fetch_latest_channel_video( $base_input, $youtube_key );
                    
                    if ( ! is_wp_error( $video_data ) ) {
                        $new_history_item = $video_data['title']; // Unique identifier
                        
                        // Skip if we already wrote about this video
                        if ( in_array( $new_history_item, $history ) ) {
                            continue; 
                        }
                        
                        $topic_to_generate = "Write a blog article summarizing and expanding on this latest video: '" . $video_data['title'] . "'. \n\nVideo Description: " . $video_data['description'];
                    }
                }
            }
            
            // B. Link Monitor (Website RSS)
            elseif ( isset( $settings['source_type'] ) && ( $settings['source_type'] === 'link_monitor' || $settings['source_type'] === 'link' ) ) {
                // Try to find feed
                $feed_url = $base_input;
                // Rudimentary check to add /feed/ if likely a standard WP site homepage
                if ( strpos( $base_input, 'feed' ) === false && strpos( $base_input, '.xml' ) === false ) {
                    $feed_url = rtrim( $base_input, '/' ) . '/feed/';
                }

                if ( function_exists( 'fetch_feed' ) ) {
                    include_once( ABSPATH . WPINC . '/feed.php' );
                    $rss = fetch_feed( $feed_url );
                    
                    if ( ! is_wp_error( $rss ) ) {
                        // Get the single latest item
                        $item = $rss->get_item( 0 ); 
                        if ( $item ) {
                            $new_history_item = $item->get_title();
                            
                            // Skip if we already wrote about this article
                            if ( in_array( $new_history_item, $history ) ) {
                                continue;
                            }

                            $topic_to_generate = "Write a unique blog article inspired by this trending piece: '" . $new_history_item . "' - Link: " . $item->get_permalink();
                        }
                    }
                }
            }
            
            // C. Standard Keyword / Web Search Trend
            else {
                // Fallback: Use the prompt directly
                $topic_to_generate = "Write an article about: " . $base_input;
                
                // Add variation instructions to avoid duplicates
                $recent_history = array_slice( $history, -5 );
                if ( ! empty( $recent_history ) ) {
                    $avoid_list = implode( ', ', $recent_history );
                    $topic_to_generate .= ". \nIMPORTANT: Do NOT repeat topics. We have already covered: [{$avoid_list}]. Find a fresh angle or specific sub-topic related to '{$base_input}'.";
                }
                
                $new_history_item = "Keyword Run: " . date( 'Y-m-d H:i:s' );
            }

            // If we couldn't find a topic (e.g., API error or duplicate), skip this job cycle
            if ( empty( $topic_to_generate ) ) {
                continue;
            }


            // --- 2. INJECT SETTINGS INTO PROMPT ---

            // Affiliate Link
            if ( ! empty( $settings['link_title'] ) && ! empty( $settings['link_url'] ) ) {
                $topic_to_generate .= "\n\nMANDATORY: Include this affiliate link naturally in the text body (around paragraph 3-5): <a href='" . esc_url( $settings['link_url'] ) . "'>" . esc_html( $settings['link_title'] ) . "</a>.";
            }

            // Custom Instructions (Tone, Forbidden words, etc.)
            if ( ! empty( $settings['custom_prompt'] ) ) {
                $topic_to_generate .= "\n\nADDITIONAL INSTRUCTIONS: " . $settings['custom_prompt'];
            }


            // --- 3. EXECUTE GENERATION ---

            // We pass the full settings array so the generator knows about Images, Audio, and Social Repurposing
            $post_id = $this->content_gen->generate_post_from_keywords_automated( $topic_to_generate, $settings );

            if ( $post_id && ! is_wp_error( $post_id ) ) {
                
                // --- 4. UPDATE DATABASE ---
                
                // Update History (Keep max 50 items to save DB space)
                $history[] = $new_history_item;
                if ( count( $history ) > 50 ) {
                    array_shift( $history );
                }

                // Calculate Next Run Time
                $interval = DAY_IN_SECONDS; // Default Daily
                if ( $job->frequency === 'weekly' ) {
                    $interval = WEEK_IN_SECONDS;
                } elseif ( $job->frequency === 'monthly' ) {
                    $interval = 30 * DAY_IN_SECONDS;
                } elseif ( $job->frequency === 'hourly' ) { // Hidden power user option?
                    $interval = HOUR_IN_SECONDS;
                }

                // Ensure next run is in the future
                $next_run_ts = strtotime( $job->next_run ) + $interval;
                if ( $next_run_ts < time() ) {
                    $next_run_ts = time() + $interval; // Catch up if server was down
                }
                $next_run = date( 'Y-m-d H:i:s', $next_run_ts );

                $wpdb->update( 
                    $table, 
                    [ 
                        'last_run'          => current_time( 'mysql' ), 
                        'next_run'          => $next_run, 
                        'articles_created'  => $job->articles_created + 1,
                        'generated_history' => json_encode( $history )
                    ],
                    [ 'id' => $job->id ]
                );
            }
        }
    }

    /**
     * Helper: Fetch the latest video details from a YouTube Channel URL.
     *
     * @param string $url The YouTube Channel URL.
     * @param string $api_key The Google API Key.
     * @return array|WP_Error ['title', 'description'] or WP_Error.
     */
    private function fetch_latest_channel_video( $url, $api_key ) {
        // 1. Try to extract Channel ID from URL (e.g., /channel/UC...)
        $channel_id = '';
        if ( preg_match( '/channel\/(UC[\w-]+)/', $url, $matches ) ) {
            $channel_id = $matches[1];
        }

        // 2. If no ID found (e.g., custom URL /c/name or handle @name), use Search API
        if ( empty( $channel_id ) ) {
            // This consumes quota but allows handles
            $search_q = urlencode( $url ); // Pass the URL or handle as query
            $search_url = "https://www.googleapis.com/youtube/v3/search?part=snippet&type=channel&q={$search_q}&key={$api_key}&maxResults=1";
            
            $s_response = wp_remote_get( $search_url );
            if ( is_wp_error( $s_response ) ) return $s_response;
            
            $s_body = json_decode( wp_remote_retrieve_body( $s_response ), true );
            if ( ! empty( $s_body['items'][0]['id']['channelId'] ) ) {
                $channel_id = $s_body['items'][0]['id']['channelId'];
            }
        }

        if ( empty( $channel_id ) ) {
            return new WP_Error( 'channel_not_found', 'Could not resolve YouTube Channel ID.' );
        }

        // 3. Get the "Uploads" Playlist ID for this channel
        // (This is cheaper on quota than searching for videos)
        $chan_url = "https://www.googleapis.com/youtube/v3/channels?part=contentDetails&id={$channel_id}&key={$api_key}";
        $c_response = wp_remote_get( $chan_url );
        if ( is_wp_error( $c_response ) ) return $c_response;

        $c_body = json_decode( wp_remote_retrieve_body( $c_response ), true );
        $uploads_playlist_id = $c_body['items'][0]['contentDetails']['relatedPlaylists']['uploads'] ?? '';

        if ( empty( $uploads_playlist_id ) ) {
            return new WP_Error( 'no_uploads', 'Channel has no uploads.' );
        }

        // 4. Get the Latest Video from the Uploads Playlist
        $v_url = "https://www.googleapis.com/youtube/v3/playlistItems?part=snippet&playlistId={$uploads_playlist_id}&maxResults=1&key={$api_key}";
        $v_response = wp_remote_get( $v_url );
        if ( is_wp_error( $v_response ) ) return $v_response;

        $v_body = json_decode( wp_remote_retrieve_body( $v_response ), true );
        if ( ! empty( $v_body['items'][0]['snippet'] ) ) {
            return [
                'title'       => $v_body['items'][0]['snippet']['title'],
                'description' => $v_body['items'][0]['snippet']['description']
            ];
        }

        return new WP_Error( 'fetch_error', 'Could not retrieve video details.' );
    }
}