<?php
/**
 * UCMR_Social_Repurposer Class
 *
 * Handles repurposing content for various social media platforms.
 * Uses AI to rewrite content and handles the publishing logic.
 *
 * @package UltimateContentMoneyRiver
 */

if ( ! defined( 'WPINC' ) ) {
    die;
}

class UCMR_Social_Repurposer {

    private $ai_client;
    private $options;

    public function __construct( $ai_client ) {
        $this->ai_client = $ai_client;
        $this->options = get_option( 'ucmr_options' );
    }

    /**
     * Main hook triggered on post publish.
     * Checks settings and triggers repurposing loop.
     *
     * @param int $post_id
     * @param WP_Post $post
     */
    public function auto_repurpose_and_post_on_publish( $post_id, $post ) {
        // Only run on publish, prevent loops
        if ( $post->post_status !== 'publish' ) return;
        if ( get_post_meta( $post_id, '_ucmr_social_repurposed', true ) ) return;
        
        $platforms = ['x', 'facebook', 'linkedin', 'instagram', 'tiktok', 'youtube'];
        
        foreach ( $platforms as $platform ) {
            // 1. Check if this platform is enabled for this specific post (e.g. from Wizard/Scheduler)
            // We check post meta first, then global settings if needed.
            // Note: The scheduler saves these preferences into the job, but when the job runs,
            // the content generator should save them as post meta so this hook sees them.
            
            $enabled_for_post = get_post_meta( $post_id, "_ucmr_repurpose_{$platform}", true );
            $enabled_global = !empty($this->options["social_{$platform}_enable"]); // Fallback if global auto-post is on
            
            // Logic: If specific meta exists, use it. If not, check global toggle.
            // For simplicity in this version, we check if the credential exists and if it was requested.
            
            if ( $enabled_for_post || ($enabled_global && !empty($this->options['social_auto_repurpose'])) ) {
                
                // 2. Check Credentials
                if ( !$this->has_credentials( $platform ) ) {
                    error_log( "[UCMR] Skipping {$platform}: Missing API credentials." );
                    continue;
                }

                // 3. Repurpose Content
                $content_data = $this->repurpose_content( $post, $platform );
                
                if ( !is_wp_error( $content_data ) ) {
                    // 4. Publish
                    $this->publish_to_social( $content_data, $platform );
                }
            }
        }
        
        // Mark as processed so we don't post again on update
        update_post_meta( $post_id, '_ucmr_social_repurposed', true );
    }

    /**
     * Generates platform-specific content using AI.
     *
     * @param int|WP_Post $post_id_or_obj
     * @param string $platform
     * @return array|WP_Error
     */
    public function repurpose_content( $post_id_or_obj, $platform ) {
        $post = get_post( $post_id_or_obj );
        if ( !$post ) return new WP_Error('invalid_post', 'Post not found');

        $title = $post->post_title;
        $text  = strip_tags( $post->post_content ); // Clean HTML
        $url   = get_permalink( $post->ID );
        
        // Affiliate Context
        $aff_link = $this->options['mon_default_link_url'] ?? '';
        $aff_text = $this->options['mon_default_link_title'] ?? 'Check this out';
        $aff_prompt = "";
        if( !empty($aff_link) ) {
            $aff_prompt = " Important: Naturally include a call to action for this link: {$aff_link} ('{$aff_text}').";
        }

        $prompt = "";
        
        switch ( $platform ) {
            case 'x':
                $prompt = "Write a viral Twitter/X thread (3-5 tweets) summarizing this article. Use emojis and hashtags. First tweet must hook the reader. Include the article link in the last tweet.{$aff_prompt}";
                break;
            case 'linkedin':
                $prompt = "Write a professional LinkedIn post summarizing this article. Use a thought-leadership tone. Use bullet points for key takeaways. End with a question to drive engagement. Include the article link.{$aff_prompt}";
                break;
            case 'facebook':
                $prompt = "Write an engaging Facebook post about this article. Use a conversational, community-focused tone. Use emojis. Include the link.{$aff_prompt}";
                break;
            case 'instagram':
                $prompt = "Write an Instagram caption for this article. Include a hook line at the top. Use line breaks. Include 10 relevant hashtags. Note: 'Link in Bio'.{$aff_prompt} Also, suggest a visual image description to go with it.";
                break;
            case 'tiktok':
                $prompt = "Write a TikTok video script (approx 60 seconds spoken) based on this article. Include: 1. Hook (visual & audio), 2. Three Value Points, 3. CTA. Keep it energetic and fast-paced.{$aff_prompt}";
                break;
            case 'youtube':
                $prompt = "Write a YouTube Video Title, Description (with timestamps/chapters based on content), and comma-separated Tags for a video version of this article.{$aff_prompt}";
                break;
        }

        $prompt .= "\n\nOriginal Article:\nTitle: $title\nContent: " . substr($text, 0, 3000); // Truncate for token limits

        $ai_result = $this->ai_client->generate_content( $prompt );
        
        if ( is_wp_error( $ai_result ) ) return $ai_result;

        return [
            'content' => $ai_result,
            'link' => $url,
            'title' => $title
        ];
    }

    /**
     * Publishes content to the API.
     *
     * @param array $content_data
     * @param string $platform
     * @return bool
     */
    public function publish_to_social( $content_data, $platform ) {
        $message = $content_data['content'];
        
        // LOGIC: Real API implementation would go here.
        // Since WordPress hosting prevents some direct API libs, we use wp_remote_post
        // This is a placeholder structure for where you would map the fields.

        $api_url = "";
        $body = [];
        $headers = [];

        switch ( $platform ) {
            case 'x':
                // Logic for X API v2
                // Requires OAuth 1.0a or OAuth 2.0 signature
                break;
            case 'facebook':
                // Logic for Graph API
                $page_id = $this->options['social_fb_app_id'] ?? ''; // Stored as ID in settings
                $token   = $this->options['social_fb_token'] ?? '';
                if($page_id && $token) {
                    $api_url = "https://graph.facebook.com/{$page_id}/feed";
                    $body = [ 'message' => $message, 'link' => $content_data['link'], 'access_token' => $token ];
                    // $this->send_remote($api_url, $body);
                }
                break;
            case 'linkedin':
                // Logic for LinkedIn UGC Post
                break;
            case 'youtube':
                // YouTube Data API logic would go here to update a video description or post to community tab
                break;
            // ... other platforms
        }

        // For now, we log success to debug.log so you can verify the AI part works
        error_log( "[UCMR Auto-Post] Platform: {$platform} | Content Preview: " . substr($message, 0, 100) . "..." );

        return true;
    }
    
    private function has_credentials( $platform ) {
        $o = $this->options;
        switch ( $platform ) {
            case 'x': return !empty($o['social_x_key']) && !empty($o['social_x_secret']);
            case 'facebook': return !empty($o['social_fb_token']);
            case 'linkedin': return !empty($o['social_li_key']);
            case 'instagram': return !empty($o['social_fb_token']); // Uses FB Graph
            case 'tiktok': return !empty($o['social_tt_key']);
            case 'youtube': return !empty($o['social_yt_key']);
            default: return false;
        }
    }

    private function send_remote($url, $body) {
        wp_remote_post($url, ['body' => $body, 'timeout' => 30]);
    }
}