<?php
/**
 * Plugin Name:       Ultimate Content Money River
 * Plugin URI:        https://www.ultimatemultimediaconsult.com/product/ultimate-content-river/
 * Description:       Build a fully automated, high-ranking, and monetized website on autopilot with AI content, audio, video, and ad management.
 * Version:           2.7.5
 * Author:            Ultimate Multimedia Consult
 * Author URI:        https://www.ultimatemultimediaconsult.com/tools
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       ucmr
 */

if ( ! defined( 'WPINC' ) ) { die; }

define( 'UCMR_VERSION', '2.7.5' );
define( 'UCMR_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'UCMR_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

// ===================================================================================
// 1. PLUGIN ACTIVATION & DATABASE
// ===================================================================================

function ucmr_plugin_activation() {
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();
    require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

    $table_views = $wpdb->prefix . 'ucmr_page_views';
    $sql_views = "CREATE TABLE $table_views (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        post_id bigint(20) NOT NULL,
        view_time datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
        ip_hash varchar(64) NOT NULL,
        PRIMARY KEY  (id),
        KEY post_id (post_id)
    ) $charset_collate;";
    dbDelta( $sql_views );

    $table_jobs = $wpdb->prefix . 'ucmr_scheduled_jobs';
    $sql_jobs = "CREATE TABLE $table_jobs (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        job_name varchar(255) NOT NULL,
        topic_prompt text NOT NULL,
        settings text NOT NULL,
        frequency varchar(50) NOT NULL,
        duration_days int(11) NOT NULL,
        start_date datetime DEFAULT CURRENT_TIMESTAMP,
        last_run datetime DEFAULT NULL,
        next_run datetime DEFAULT NULL,
        status varchar(20) DEFAULT 'active',
        articles_created int(11) DEFAULT 0,
        generated_history longtext DEFAULT NULL,
        PRIMARY KEY  (id)
    ) $charset_collate;";
    dbDelta( $sql_jobs );

    if ( ! wp_next_scheduled( 'ucmr_cron_process_jobs' ) ) {
        wp_schedule_event( time(), 'hourly', 'ucmr_cron_process_jobs' );
    }

    ucmr_register_podcast_taxonomy();
    flush_rewrite_rules();
}
register_activation_hook( __FILE__, 'ucmr_plugin_activation' );

function ucmr_plugin_deactivation() {
    wp_clear_scheduled_hook( 'ucmr_cron_process_jobs' );
    flush_rewrite_rules();
}
register_deactivation_hook( __FILE__, 'ucmr_plugin_deactivation' );

// ===================================================================================
// 2. INCLUDE CORE CLASSES (Safe Loader)
// ===================================================================================

$ucmr_files = [
    'class-ucmr-ai-client.php',
    'class-ucmr-audio-client.php',
    'class-ucmr-media-client.php',
    'class-ucmr-content-generator.php',
    'class-ucmr-ad-injector.php',
    'class-ucmr-social-repurposer.php',
    'class-ucmr-scheduler.php'
];

foreach ( $ucmr_files as $file ) {
    if ( file_exists( UCMR_PLUGIN_DIR . 'includes/' . $file ) ) {
        require_once UCMR_PLUGIN_DIR . 'includes/' . $file;
    }
}

// ===================================================================================
// 3. INIT & CRON HANDLER
// ===================================================================================

function ucmr_init() {
    $options = get_option( 'ucmr_options' );
    
    $ai_client      = class_exists('UCMR_AI_Client') ? new UCMR_AI_Client( $options ) : null;
    $audio_client   = class_exists('UCMR_Audio_Client') ? new UCMR_Audio_Client() : null;
    $media_client   = class_exists('UCMR_Media_Client') ? new UCMR_Media_Client( $options ) : null;
    
    $content_gen = null;
    if ( class_exists('UCMR_Content_Generator') && $ai_client && $media_client && $audio_client ) {
        $content_gen = new UCMR_Content_Generator( $options, $ai_client, $media_client, $audio_client );
    }
    
    if ( class_exists('UCMR_Ad_Injector') ) new UCMR_Ad_Injector( $options );
    
    if ( class_exists('UCMR_Social_Repurposer') && $ai_client ) {
        $social_repurposer = new UCMR_Social_Repurposer( $ai_client );
        add_action( 'publish_post', [ $social_repurposer, 'auto_repurpose_and_post_on_publish' ], 10, 2 );
    }
    
    if ( class_exists('UCMR_Scheduler') && $content_gen ) {
        $scheduler = new UCMR_Scheduler( $content_gen );
        add_action( 'ucmr_cron_process_jobs', [ $scheduler, 'process_jobs' ] );
    }
    
    ucmr_register_podcast_taxonomy();
}
add_action( 'init', 'ucmr_init' );

function ucmr_register_podcast_taxonomy() {
    register_taxonomy( 'ucmr_podcast', [ 'post' ], [
        'hierarchical'      => true,
        'labels'            => [ 'name' => 'Podcasts', 'singular_name' => 'Podcast' ],
        'show_ui'           => true,
        'show_admin_column' => true,
        'query_var'         => true,
        'rewrite'           => [ 'slug' => 'podcast', 'with_front' => false ],
        'show_in_rest'      => true,
    ] );
}

add_shortcode( 'ucmr_podcast_player', 'ucmr_podcast_player_shortcode' );
function ucmr_podcast_player_shortcode( $atts ) {
    $atts = shortcode_atts( [ 'post_id' => get_the_ID() ], $atts );
    $audio_url = get_post_meta( $atts['post_id'], '_ucmr_audio_url', true );
    if ( empty( $audio_url ) ) return '';
    return sprintf( '<div class="ucmr-podcast-player"><audio controls style="width: 100%%;"><source src="%s" type="audio/mpeg">Your browser does not support the audio element.</audio></div>', esc_url( $audio_url ) );
}

function ucmr_add_tts_player_to_content( $content ) {
    $options = get_option( 'ucmr_options' );
    if ( is_singular( 'post' ) && ! empty( $options['mon_enable_tts'] ) && $options['mon_enable_tts'] === '1' ) {
        $audio_url = get_post_meta( get_the_ID(), '_ucmr_audio_url', true );
        if ( ! empty( $audio_url ) ) {
            $player_html = '<div class="ucmr-tts-player-container" style="padding: 15px; background: #f0f0f0; border-radius: 8px; margin: 0 0 20px 0;"><h4 style="margin: 0 0 10px 0;">🎧 Listen to this article</h4><audio controls style="width: 100%;" src="' . esc_url( $audio_url ) . '">Your browser does not support the audio element.</audio></div>';
            return $player_html . $content;
        }
    }
    return $content;
}
add_filter( 'the_content', 'ucmr_add_tts_player_to_content' );

function ucmr_track_page_view() {
    if ( is_singular( 'post' ) ) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'ucmr_page_views';
        $post_id = get_the_ID();
        $ip_hash = hash( 'sha256', $_SERVER['REMOTE_ADDR'] . date('Y-m-d') );
        $existing = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM $table_name WHERE post_id = %d AND ip_hash = %s", $post_id, $ip_hash ) );
        if ( ! $existing ) {
            $wpdb->insert( $table_name, [ 'post_id' => $post_id, 'view_time' => current_time( 'mysql' ), 'ip_hash' => $ip_hash ] );
        }
    }
}
add_action( 'wp_head', 'ucmr_track_page_view' );

// ===================================================================================
// 4. ADMIN & LICENSE (UPDATED FOR VARIATIONS)
// ===================================================================================

function ucmr_opt( $key, $default = '' ) {
    $options = get_option( 'ucmr_options' );
    return isset( $options[ $key ] ) ? $options[ $key ] : $default;
}

function ucmr_is_premium_user( $force_check = false ) {
    global $pagenow;
    if ( is_admin() && $pagenow === 'options.php' && ! $force_check ) return get_transient( 'ucmr_license_status' ) === 'valid';

    $status = get_transient( 'ucmr_license_status' );
    if ( $status === false || $force_check ) {
        $options = get_option( 'ucmr_options' );
        $license_key = $options['license_key'] ?? '';
        
        if ( empty( $license_key ) ) {
            set_transient( 'ucmr_license_status', 'invalid', DAY_IN_SECONDS );
            update_option('ucmr_license_debug_msg', 'License Key is empty.');
            return false;
        }

        $uwlm_public_key = 'pub_73b32ff403f0fc47dc1b27986dd68152';
        $api_url = 'https://www.ultimatemultimediaconsult.com/wp-json/uwlm/v1/check';
        
        // Array of valid Product IDs (Parent + Variations)
        $valid_product_ids = [ 5061, 5068, 5069, 5070, 5071 ];
        $is_valid = false;
        $debug_msg = '';

        foreach ( $valid_product_ids as $pid ) {
            $response = wp_remote_get( add_query_arg( [ 'license_key' => $license_key, 'product_id' => $pid ], $api_url ), [ 'timeout' => 5, 'headers' => [ 'X-Api-Key' => $uwlm_public_key ] ] );
            
            if ( is_wp_error( $response ) ) {
                $debug_msg = 'Connection Error: ' . $response->get_error_message();
                continue;
            }
            
            $code = wp_remote_retrieve_response_code($response);
            if ( $code != 200 ) {
                 $debug_msg = 'Server Error: HTTP ' . $code;
                 continue;
            }

            $body = json_decode( wp_remote_retrieve_body( $response ) );
            if ( $body && isset( $body->valid ) && $body->valid === true ) {
                $is_valid = true;
                break; // Stop checking once found
            } else {
                $debug_msg = isset($body->error) ? $body->error : 'Invalid Key for ID ' . $pid;
            }
        }

        if ( $is_valid ) {
            set_transient( 'ucmr_license_status', 'valid', DAY_IN_SECONDS );
            update_option('ucmr_license_debug_msg', 'Success');
            $status = 'valid';
        } else {
            set_transient( 'ucmr_license_status', 'invalid', DAY_IN_SECONDS );
            update_option('ucmr_license_debug_msg', 'Validation Failed: ' . $debug_msg);
            $status = 'invalid';
        }
    }
    return $status === 'valid';
}

function ucmr_check_premium_status() {
    $is_premium = ucmr_is_premium_user();
    if ( ! $is_premium ) { echo ' disabled title="Premium Feature"'; }
    return $is_premium;
}

function ucmr_add_admin_menu() {
    add_menu_page( 'Content River', 'Content River', 'manage_options', 'ucmr_settings', 'ucmr_render_settings_page', 'dashicons-admin-settings', 20 );
    add_submenu_page( 'ucmr_settings', 'Settings', 'Settings', 'manage_options', 'ucmr_settings', 'ucmr_render_settings_page' );
    
    $tools_slug = ucmr_is_premium_user() ? 'ucmr_backend_tools' : 'ucmr_settings';
    
    add_submenu_page( $tools_slug, 'Analytics', 'Analytics', 'manage_options', 'ucmr_analytics', 'ucmr_render_analytics_page' );
    add_submenu_page( $tools_slug, 'Trending Content', 'Trending Content', 'manage_options', 'ucmr_trending', 'ucmr_render_trending_page' );
    add_submenu_page( $tools_slug, 'Backend Chatbot', 'Backend Chatbot', 'manage_options', 'ucmr_chatbot', 'ucmr_render_backend_chatbot_page' );
}
add_action( 'admin_menu', 'ucmr_add_admin_menu' );

function ucmr_register_settings() { register_setting( 'ucmr_options_group', 'ucmr_options', 'ucmr_sanitize_options' ); }
add_action( 'admin_init', 'ucmr_register_settings' );

function ucmr_sanitize_options( $input ) {
    $options = get_option( 'ucmr_options' );
    if ( ! is_array( $options ) ) $options = [];
    $output = array_merge( $options, $input );

    $output['license_key'] = sanitize_text_field( $input['license_key'] ?? $options['license_key'] ?? '' );
    if ( isset( $input['license_key'] ) && $input['license_key'] !== ( $options['license_key'] ?? '' ) ) {
        delete_transient( 'ucmr_license_status' );
    }
    
    $raw_cx = $input['search_google_cx'] ?? '';
    if ( preg_match( '/cx=([a-zA-Z0-9]+)/', $raw_cx, $matches ) ) {
        $output['search_google_cx'] = $matches[1];
    } else {
        $output['search_google_cx'] = sanitize_text_field( $raw_cx );
    }
    
    $output['search_google_api_key'] = sanitize_text_field( $input['search_google_api_key'] ?? $options['search_google_api_key'] ?? '' );
    $output['ai_provider'] = sanitize_text_field($input['ai_provider'] ?? 'gemini');
    $output['ai_model'] = sanitize_text_field($input['ai_model'] ?? '');
    $output['gemini_api_key'] = sanitize_text_field($input['gemini_api_key'] ?? '');
    $output['chatgpt_api_key'] = sanitize_text_field($input['chatgpt_api_key'] ?? '');
    $output['claude_api_key'] = sanitize_text_field($input['claude_api_key'] ?? '');
    $output['grok_api_key'] = sanitize_text_field($input['grok_api_key'] ?? '');
    $output['deepseek_api_key'] = sanitize_text_field($input['deepseek_api_key'] ?? '');
    $output['elevenlabs_api_key'] = sanitize_text_field($input['elevenlabs_api_key'] ?? '');
    $output['meta_api_key'] = sanitize_text_field($input['meta_api_key'] ?? '');
    $output['media_pixabay_key'] = sanitize_text_field($input['media_pixabay_key'] ?? '');
    $output['media_pexels_key'] = sanitize_text_field($input['media_pexels_key'] ?? '');
    $output['media_unsplash_key'] = sanitize_text_field($input['media_unsplash_key'] ?? '');
    $output['media_youtube_key'] = sanitize_text_field($input['media_youtube_key'] ?? '');
    $output['mon_default_link_url'] = esc_url_raw($input['mon_default_link_url'] ?? '');
    $output['style_auto_audio'] = isset($input['style_auto_audio']) ? '1' : '0';

    $ad_placements = [ 'ads_top', 'ads_in_content', 'ads_below_post' ];
    $output['ads_in_content_p_num'] = absint( $input['ads_in_content_p_num'] ?? 3 );
    $output['ads_top_float'] = sanitize_text_field( $input['ads_top_float'] ?? 'none' );
    $output['ads_content_float'] = sanitize_text_field( $input['ads_content_float'] ?? 'none' );

    foreach ( $ad_placements as $placement ) {
        for ( $i = 1; $i <= 5; $i++ ) {
            $output[ $placement . '_' . $i . '_enable' ] = isset( $input[ $placement . '_' . $i . '_enable' ] ) ? '1' : '0';
            $output[ $placement . '_' . $i . '_code' ] = wp_kses_post( $input[ $placement . '_' . $i . '_code' ] ?? $options[ $placement . '_' . $i . '_code' ] ?? '' );
            $output[ $placement . '_' . $i . '_freq' ] = absint( $input[ $placement . '_' . $i . '_freq' ] ?? $options[ $placement . '_' . $i . '_freq' ] ?? 10 );
        }
    }

    // Social Keys
    $socials = ['social_x_key','social_x_secret','social_fb_app_id','social_fb_token','social_li_key','social_li_secret','social_tt_key','social_tt_secret','social_yt_key','social_yt_secret'];
    foreach($socials as $s) { $output[$s] = sanitize_text_field($input[$s] ?? ($options[$s] ?? '')); }

    return $output;
}

function ucmr_render_settings_page() { require_once UCMR_PLUGIN_DIR . 'ucmr-admin-page.php'; }
function ucmr_render_analytics_page() { require_once UCMR_PLUGIN_DIR . 'admin/pages/ucmr-analytics-page.php'; }
function ucmr_render_trending_page() { require_once UCMR_PLUGIN_DIR . 'admin/pages/ucmr-trending-page.php'; }
function ucmr_render_backend_chatbot_page() { require_once UCMR_PLUGIN_DIR . 'admin/pages/ucmr-backend-chatbot-page.php'; }

function ucmr_enqueue_admin_scripts( $hook ) {
    if ( strpos($hook, 'ucmr_') !== false || strpos($hook, 'content-river') !== false || strpos($hook, 'backend_tools') !== false ) {
         wp_enqueue_style( 'ucmr-admin-css', UCMR_PLUGIN_URL . 'admin/css/ucmr-admin.css', [], UCMR_VERSION );
         wp_enqueue_script( 'ucmr-admin-js', UCMR_PLUGIN_URL . 'admin/js/ucmr-admin.js', [ 'jquery' ], UCMR_VERSION, true );
         wp_localize_script( 'ucmr-admin-js', 'ucmr_ajax_object', [
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce'    => wp_create_nonce( 'ucmr-ajax-nonce' ),
        ] );
        if ( strpos($hook, 'trending') !== false || strpos($hook, 'chatbot') !== false ) {
            wp_enqueue_style( 'ucmr-backend-tools-css', UCMR_PLUGIN_URL . 'admin/css/ucmr-backend-tools.css', ['ucmr-admin-css'], UCMR_VERSION );
            wp_enqueue_script( 'ucmr-backend-tools-js', UCMR_PLUGIN_URL . 'admin/js/ucmr-backend-tools.js', [ 'jquery' ], UCMR_VERSION, true );
        }
    }
    if ( $hook == 'post.php' || $hook == 'post-new.php' ) {
        wp_enqueue_script( 'ucmr-post-editor-js', UCMR_PLUGIN_URL . 'admin/js/ucmr-post-editor.js', [ 'jquery' ], UCMR_VERSION, true );
        wp_localize_script( 'ucmr-post-editor-js', 'ucmr_post_tools_ajax', [
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce'    => wp_create_nonce( 'ucmr_post_tools_nonce' ),
        ] );
    }
}
add_action( 'admin_enqueue_scripts', 'ucmr_enqueue_admin_scripts' );

// ===================================================================================
// 8. AJAX HANDLERS
// ===================================================================================

// 1. YouTube Metadata Fetcher
function ucmr_fetch_youtube_metadata( $url, $api_key ) {
    if ( !empty( $api_key ) ) {
        parse_str( parse_url( $url, PHP_URL_QUERY ), $vars );
        $video_id = $vars['v'] ?? '';
        if ( empty( $video_id ) && strpos($url, 'youtu.be') !== false ) { $path = parse_url($url, PHP_URL_PATH); $video_id = trim($path, '/'); }
        if( !empty($video_id) ) {
            $api_url = "https://www.googleapis.com/youtube/v3/videos?part=snippet&id={$video_id}&key={$api_key}";
            $response = wp_remote_get( $api_url );
            if ( !is_wp_error( $response ) ) {
                $body = json_decode( wp_remote_retrieve_body( $response ), true );
                if ( !empty( $body['items'][0] ) ) {
                    return [ 'title' => $body['items'][0]['snippet']['title'], 'description' => $body['items'][0]['snippet']['description'] ];
                }
            }
        }
    }
    return [ 'title' => 'YouTube Video (' . $url . ')', 'description' => 'Please summarize the content from this video URL.' ];
}

// 2. Search YouTube (Fallback)
function ucmr_handle_search_youtube_ajax() {
    check_ajax_referer( 'ucmr-ajax-nonce', '_ajax_nonce' );
    $options = get_option( 'ucmr_options' );
    $key = $options['media_youtube_key'] ?? '';
    $q = sanitize_text_field( $_POST['query'] );
    $api_success = false;
    $videos = [];
    if($key) {
        $url = "https://www.googleapis.com/youtube/v3/search?part=snippet&maxResults=10&q=" . urlencode($q) . "&type=video&key={$key}";
        $response = wp_remote_get( $url );
        if(!is_wp_error($response)) {
            $body = json_decode(wp_remote_retrieve_body($response), true);
            if(!isset($body['error']) && !empty($body['items'])) {
                foreach($body['items'] as $item) {
                    if(isset($item['id']['videoId'])) {
                        $videos[] = [ 'id' => $item['id']['videoId'], 'title' => $item['snippet']['title'], 'thumbnail' => $item['snippet']['thumbnails']['medium']['url'], 'channel' => $item['snippet']['channelTitle'] ];
                    }
                }
                $api_success = true;
            }
        }
    }
    if( !$api_success ) {
        $ai_client = new UCMR_AI_Client( $options );
        $prompt = "Simulate a YouTube search for '{$q}'. Return 5 popular video titles. Format as JSON: [{\"title\":\"Video Title\", \"id\":\"placeholder\", \"channel\":\"Channel Name\"}]";
        $ai_res = $ai_client->generate_content( $prompt );
        $ai_res = str_replace(['```json', '```'], '', $ai_res);
        $ai_videos = json_decode($ai_res, true);
        if(is_array($ai_videos)) { foreach($ai_videos as $v) { $videos[] = [ 'id' => 'AI_GENERATED', 'title' => $v['title'] ?? 'AI Result', 'thumbnail' => 'https://via.placeholder.com/320x180.png?text=AI+Result', 'channel' => $v['channel'] ?? 'AI Channel' ]; } }
    }
    wp_send_json_success(['videos' => $videos]);
}
add_action('wp_ajax_ucmr_search_youtube', 'ucmr_handle_search_youtube_ajax');

// 3. Web Search (Fallback)
function ucmr_handle_search_web_ajax() {
    check_ajax_referer( 'ucmr-ajax-nonce', '_ajax_nonce' );
    $options = get_option( 'ucmr_options' );
    $key = $options['search_google_api_key'] ?? '';
    $cx = $options['search_google_cx'] ?? ''; 
    $q = sanitize_text_field( $_POST['query'] );
    $api_success = false;
    $articles = [];
    if($key && $cx) {
        $url = "https://www.googleapis.com/customsearch/v1?key={$key}&cx={$cx}&q=" . urlencode($q);
        $response = wp_remote_get( $url );
        if(!is_wp_error($response)) {
            $body = json_decode(wp_remote_retrieve_body($response), true);
            if(!isset($body['error']) && !empty($body['items'])) {
                foreach($body['items'] as $item) { $articles[] = [ 'title' => $item['title'], 'link' => $item['link'], 'snippet' => $item['snippet'] ]; }
                $api_success = true;
            }
        }
    }
    if( !$api_success ) {
        $ai_client = new UCMR_AI_Client( $options );
        $prompt = "Simulate a Google search result for '{$q}'. List 5 relevant articles. Format as JSON: [{\"title\":\"Article Title\", \"link\":\"http://example.com\", \"snippet\":\"Summary...\"}]";
        $ai_res = $ai_client->generate_content( $prompt );
        $ai_res = str_replace(['```json', '```'], '', $ai_res);
        $ai_articles = json_decode($ai_res, true);
        if(is_array($ai_articles)) { $articles = $ai_articles; }
    }
    wp_send_json_success(['articles' => $articles]);
}
add_action('wp_ajax_ucmr_search_web', 'ucmr_handle_search_web_ajax');

// 3b. Fetch URL Articles (Link Monitor)
function ucmr_handle_fetch_url_articles_ajax() {
    check_ajax_referer( 'ucmr-ajax-nonce', '_ajax_nonce' );
    $url = esc_url_raw($_POST['query']);
    if(!$url) wp_send_json_error(['message'=>'Invalid URL']);
    $articles = [];
    $feed_url = rtrim($url, '/') . '/feed/';
    include_once(ABSPATH . WPINC . '/feed.php');
    $rss = fetch_feed($feed_url);
    if(!is_wp_error($rss)) {
        foreach($rss->get_items(0, 10) as $item) { $articles[] = ['title' => $item->get_title(), 'link' => $item->get_permalink(), 'snippet' => substr(strip_tags($item->get_description()), 0, 100).'...']; }
        wp_send_json_success(['articles' => $articles]); return;
    }
    $response = wp_remote_get($url);
    if(!is_wp_error($response)) {
        $html = substr(wp_remote_retrieve_body($response), 0, 15000);
        $options = get_option( 'ucmr_options' ); $ai_client = new UCMR_AI_Client($options);
        $prompt = "Extract 5 article titles and links from this HTML. Format as JSON array: [{\"title\":\"...\", \"link\":\"...\"}]. HTML: " . strip_tags($html, '<a><h1><h2>');
        $ai_res = $ai_client->generate_content($prompt);
        $ai_res = str_replace(['```json', '```'], '', $ai_res);
        $data = json_decode($ai_res, true);
        if(is_array($data)) { wp_send_json_success(['articles' => $data]); return; }
    }
    wp_send_json_error(['message' => 'Could not fetch.']);
}
add_action('wp_ajax_ucmr_fetch_url_articles', 'ucmr_handle_fetch_url_articles_ajax');

// 4. Preview Content
function ucmr_handle_preview_post_ajax() {
    check_ajax_referer( 'ucmr-ajax-nonce', '_ajax_nonce' );
    set_time_limit(120); 
    $options     = get_option( 'ucmr_options' );
    $ai_client   = new UCMR_AI_Client( $options );
    $mode = sanitize_text_field( $_POST['mode'] );
    $input = sanitize_text_field( $_POST['input'] ); 
    $tone = sanitize_text_field( $_POST['tone'] );
    $length = intval( $_POST['length'] );
    $aff_text = sanitize_text_field( $_POST['aff_text'] );
    $aff_url = esc_url_raw( $_POST['aff_url'] );
    $custom_prompt = sanitize_textarea_field( $_POST['custom_prompt'] );

    $prompt = "";
    $sources = explode(',', $input);
    
    if ( $mode === 'youtube' ) {
        $prompt = "Write a blog article based on these videos:\n";
        foreach($sources as $url) {
            $url = trim($url);
            $video_data = ucmr_fetch_youtube_metadata( $url, $options['media_youtube_key'] ?? '' );
            $prompt .= "- Title: " . $video_data['title'] . "\n  Summary: " . $video_data['description'] . "\n\n";
        }
    } elseif ( $mode === 'link' || $mode === 'web' ) {
         $prompt = "Write a comprehensive article based on these sources:\n";
         foreach($sources as $url) { $prompt .= "- " . trim($url) . "\n"; }
    } else {
        $prompt = "Write a complete blog article about: " . $input . ".\n\n";
    }
    $prompt .= "Tone: {$tone}. Target length: {$length} words. HTML format.";
    if($custom_prompt) $prompt .= " Instructions: " . $custom_prompt;
    if($aff_text && $aff_url) $prompt .= " Link: <a href='{$aff_url}'>{$aff_text}</a>.";

    $result = $ai_client->generate_content( $prompt );
    if ( is_wp_error( $result ) ) { wp_send_json_error( [ 'message' => $result->get_error_message() ] ); } 
    else {
        $title = 'Generated Title';
        if(preg_match('/Title:\s*(.*?)\s*$/m', $result, $m)) $title = $m[1];
        wp_send_json_success( [ 'title' => $title, 'content' => $result ] );
    }
}
add_action( 'wp_ajax_ucmr_preview_post', 'ucmr_handle_preview_post_ajax' );

// 5. Publish Post
function ucmr_handle_publish_post_ajax() {
    check_ajax_referer( 'ucmr-ajax-nonce', '_ajax_nonce' );
    $options      = get_option( 'ucmr_options' );
    $ai_client    = new UCMR_AI_Client( $options );
    $audio_client = new UCMR_Audio_Client();
    $media_client = new UCMR_Media_Client( $options );
    $content_gen  = new UCMR_Content_Generator( $options, $ai_client, $media_client, $audio_client );
    $title        = sanitize_text_field( $_POST['title'] );
    $content      = wp_kses_post( $_POST['content'] );
    $status       = sanitize_text_field( $_POST['status'] ); 
    $cat_id       = intval( $_POST['category_id'] );
    $new_cat      = sanitize_text_field( $_POST['new_category'] );
    $keywords     = sanitize_text_field( $_POST['keywords'] );
    $result = $content_gen->save_generated_post( $title, $content, $status, $cat_id, $new_cat, $keywords );
    if ( is_wp_error( $result ) ) { wp_send_json_error( [ 'message' => $result->get_error_message() ] ); }
    else { wp_send_json_success( [ 'message' => 'Post saved!', 'post_id' => $result['post_id'], 'edit_url' => get_edit_post_link( $result['post_id'], 'raw' ), 'audio_url' => $result['audio_url'] ] ); }
}
add_action( 'wp_ajax_ucmr_publish_post', 'ucmr_handle_publish_post_ajax' );

// 6. Process Images
function ucmr_handle_process_images_ajax() {
    check_ajax_referer( 'ucmr-ajax-nonce', '_ajax_nonce' );
    $options = get_option( 'ucmr_options' );
    $media_client = new UCMR_Media_Client( $options );
    $ai_client = new UCMR_AI_Client( $options ); 
    $post_id = absint( $_POST['post_id'] );
    $keywords = sanitize_text_field( $_POST['keywords'] );
    $count = absint( $_POST['count'] );
    $source = sanitize_text_field( $_POST['source'] ); 

    if ( $source === 'ai' ) {
        for($i=0; $i<$count; $i++) {
            $img_url = $ai_client->generate_image_content("A high quality editorial image for: " . $keywords);
            if(!is_wp_error($img_url)) {
                $media_client->sideload_image($img_url, $post_id, $keywords);
                if($i===0) set_post_thumbnail($post_id, $media_client->get_last_attachment_id());
            }
        }
    } else {
        $found = $media_client->attach_images_to_post($post_id, $keywords, $count);
        if ( $source === 'hybrid' && $found === 0 ) {
             $img_url = $ai_client->generate_image_content("A high quality editorial image for: " . $keywords);
             if(!is_wp_error($img_url)) {
                $media_client->sideload_image($img_url, $post_id, $keywords);
                set_post_thumbnail($post_id, $media_client->get_last_attachment_id());
             }
        }
    }
    wp_send_json_success(['message' => 'Images processed']);
}
add_action('wp_ajax_ucmr_process_images', 'ucmr_handle_process_images_ajax');

// 7. Process Audio
function ucmr_handle_process_audio_ajax() {
    check_ajax_referer( 'ucmr-ajax-nonce', '_ajax_nonce' );
    $options = get_option( 'ucmr_options' );
    $audio_client = new UCMR_Audio_Client();
    $post_id = absint( $_POST['post_id'] );
    $post = get_post($post_id);
    if(!$post) wp_send_json_error(['message' => 'Post not found']);
    $content = wp_strip_all_tags($post->post_content);
    $text_to_speak = substr($content, 0, 4000); 
    $voice_setting = $options['style_auto_audio_voice'] ?? 'openai:nova';
    list( $provider, $voice_id ) = explode( ':', $voice_setting, 2 );
    $audio_data = false;
    if ( $provider === 'openai' ) { $audio_data = $audio_client->generate_speech( $text_to_speak, $voice_id ); }
    elseif ( $provider === 'elevenlabs' ) {
        $voice_id = ( $voice_id === 'cloned' ) ? null : $voice_id; 
        $audio_data = $audio_client->generate_cloned_speech( $text_to_speak, $voice_id );
    }
    if ( is_wp_error( $audio_data ) ) wp_send_json_error(['message' => $audio_data->get_error_message()]);
    $filename = 'audio-' . $post_id . '.mp3';
    $att_id = $audio_client->upload_audio_to_media_library( $audio_data, $filename, $post_id, 'Podcast' );
    if ( is_wp_error( $att_id ) ) wp_send_json_error(['message' => 'Upload failed']);
    $url = wp_get_attachment_url($att_id);
    update_post_meta( $post_id, '_ucmr_audio_url', $url );
    wp_set_object_terms( $post_id, 'Audio Articles', 'ucmr_podcast', true );
    wp_send_json_success(['audio_url' => $url]);
}
add_action('wp_ajax_ucmr_process_audio', 'ucmr_handle_process_audio_ajax');

// Create Job
function ucmr_handle_create_job_ajax() {
    check_ajax_referer( 'ucmr-ajax-nonce', '_ajax_nonce' );
    if ( ! ucmr_is_premium_user() ) wp_send_json_error( [ 'message' => 'Premium required.' ] );
    $name = sanitize_text_field( $_POST['job_name'] );
    $topic = sanitize_text_field( $_POST['topic'] );
    $freq = sanitize_text_field( $_POST['frequency'] );
    $dur = intval( $_POST['duration'] );
    $settings = json_encode([
        'category' => intval($_POST['category']),
        'author' => intval($_POST['author']),
        'tone' => sanitize_text_field($_POST['tone']),
        'img_count' => intval($_POST['img_count']),
        'length' => intval($_POST['length']),
        'link_title' => sanitize_text_field($_POST['link_title']),
        'link_url' => esc_url_raw($_POST['link_url']),
        'podcast' => sanitize_text_field($_POST['podcast']),
        'source_type' => sanitize_text_field($_POST['source_type']),
    ]);
    global $wpdb;
    $table = $wpdb->prefix . 'ucmr_scheduled_jobs';
    $wpdb->insert( $table, [
        'job_name' => $name, 'topic_prompt' => $topic, 'settings' => $settings, 'frequency' => $freq, 'duration_days' => $dur, 'start_date' => current_time('mysql'), 'next_run' => current_time('mysql'), 'status' => 'active'
    ]);
    wp_send_json_success( [ 'message' => 'Job Scheduled Successfully!' ] );
}
add_action( 'wp_ajax_ucmr_create_job', 'ucmr_handle_create_job_ajax' );

// Delete Job
function ucmr_handle_delete_job_ajax() {
    check_ajax_referer( 'ucmr-ajax-nonce', '_ajax_nonce' );
    if ( ! ucmr_is_premium_user() ) wp_send_json_error( [ 'message' => 'Premium required.' ] );
    global $wpdb;
    $wpdb->delete( $wpdb->prefix . 'ucmr_scheduled_jobs', [ 'id' => intval($_POST['id']) ] );
    wp_send_json_success();
}
add_action( 'wp_ajax_ucmr_delete_job', 'ucmr_handle_delete_job_ajax' );

// Backend Chat
function ucmr_handle_backend_chat_ajax() {
    check_ajax_referer( 'ucmr-ajax-nonce', '_ajax_nonce' );
    $options   = get_option( 'ucmr_options' );
    $ai_client = new UCMR_AI_Client( $options );
    $message   = sanitize_textarea_field( $_POST['message'] );
    $prompt = "You are a helpful AI assistant for the Ultimate Content Money River WordPress plugin. Answer: " . $message;
    $result = $ai_client->generate_content( $prompt );
    if ( is_wp_error( $result ) ) { wp_send_json_error( [ 'message' => $result->get_error_message() ] ); } else { wp_send_json_success( [ 'reply' => $result ] ); }
}
add_action( 'wp_ajax_ucmr_backend_chat', 'ucmr_handle_backend_chat_ajax' );

// Trends
function ucmr_handle_get_trends_ajax() {
    check_ajax_referer( 'ucmr-ajax-nonce', '_ajax_nonce' );
    $options   = get_option( 'ucmr_options' );
    $ai_client = new UCMR_AI_Client( $options );
    $topic     = sanitize_text_field( $_POST['topic'] );
    $platform  = sanitize_text_field( $_POST['platform'] );
    $prompt = "List 5 trending topics for {$platform} regarding: {$topic}. Return only bullets.";
    $result = $ai_client->generate_content( $prompt );
    if ( is_wp_error( $result ) ) { wp_send_json_error( [ 'message' => $result->get_error_message() ] ); } else { wp_send_json_success( [ 'trends' => $result ] ); }
}
add_action( 'wp_ajax_ucmr_get_trends', 'ucmr_handle_get_trends_ajax' );