<?php
/**
 * Admin Actions Class
 *
 * Handles processing admin forms and actions.
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class UWLM_Admin_Actions {

	/**
	 * Hook in.
	 * --- THIS IS THE FIX ---
	 * We hook actions to 'admin_init' instead of running them directly.
	 */
	public static function init() {
		// AJAX for Pro License activation
		add_action( 'wp_ajax_uwlm_activate_pro_license', array( __CLASS__, 'ajax_activate_pro_license' ) );
		add_action( 'wp_ajax_uwlm_deactivate_pro_license', array( __CLASS__, 'ajax_deactivate_pro_license' ) );
		
		// AJAX for generating a key on "Add New" page
		add_action( 'wp_ajax_uwlm_generate_key_string', array( __CLASS__, 'ajax_generate_key_string' ) );
		
		// --- FIX ---
		// Hook all form handlers to admin_init
		add_action( 'admin_init', array( __CLASS__, 'handle_add_new_license' ) );
		add_action( 'admin_init', array( __CLASS__, 'handle_import_export_actions' ) );
		add_action( 'admin_init', array( __CLASS__, 'handle_license_key_table_actions' ) );
	}
	
	/**
	 * Handle the form submission for adding a new license.
	 */
	public static function handle_add_new_license() {
		if ( ! isset( $_POST['action'] ) || $_POST['action'] !== 'uwlm_add_new_license' ) {
			return;
		}
		
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'uwlm_add_new' ) ) {
			wp_die( 'Nonce verification failed.' );
		}
		
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( 'You do not have permission.' );
		}

		$key_string = isset( $_POST['license_key'] ) ? sanitize_text_field( $_POST['license_key'] ) : '';
		$product_id = isset( $_POST['product_id'] ) ? absint( $_POST['product_id'] ) : 0;
		$max_activations = isset( $_POST['max_activations'] ) ? absint( $_POST['max_activations'] ) : 1;
		$expires_at = isset( $_POST['expires_at'] ) && ! empty( $_POST['expires_at'] ) ? sanitize_text_field( $_POST['expires_at'] ) : null;
		
		if ( empty( $key_string ) || empty( $product_id ) ) {
			add_action( 'admin_notices', function() {
				echo '<div class="notice notice-error"><p>' . esc_html__( 'License Key and Product are required.', 'uwlm' ) . '</p></div>';
			});
			return;
		}

		// Determine if it's a variation
		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			add_action( 'admin_notices', function() {
				echo '<div class="notice notice-error"><p>' . esc_html__( 'Selected product not found.', 'uwlm' ) . '</p></div>';
			});
			return;
		}
		
		$parent_id = $product->is_type( 'variation' ) ? $product->get_parent_id() : $product_id;
		$variation_id = $product->is_type( 'variation' ) ? $product_id : 0;

		// Calculate validity days from expires_at
		$validity_days = 0;
		if ( $expires_at ) {
			$today = new DateTime();
			$expiry = new DateTime( $expires_at );
			$diff = $today->diff( $expiry );
			$validity_days = $diff->days > 0 ? $diff->days : 0;
		} else {
			$expires_at = null; // Ensure it's null for lifetime
		}

		$result = self::insert_manual_key( $key_string, $parent_id, $variation_id, $max_activations, $validity_days, $expires_at );
		
		if ( $result ) {
			add_action( 'admin_notices', function() {
				$list_url = admin_url( 'admin.php?page=uwlm' );
				echo '<div class="notice notice-success"><p>' . wp_kses_post( sprintf( __( 'License key created successfully. <a href="%s">View all keys</a>', 'uwlm' ), esc_url( $list_url ) ) ) . '</p></div>';
			});
		} else {
			add_action( 'admin_notices', function() {
				echo '<div class="notice notice-error"><p>' . esc_html__( 'Error: Could not create license key. It might already exist.', 'uwlm' ) . '</p></div>';
			});
		}
	}

	/**
	 * Helper function to insert a manual key.
	 * @return bool|int Insert ID or false on failure.
	 */
	public static function insert_manual_key( $key_string, $product_id, $variation_id, $max_activations, $validity_days, $expires_at, $status = 'active', $customer_id = 0, $order_id = 0, $activations = 0 ) {
		global $wpdb;
		$table = $wpdb->prefix . UWLM_LICENSE_KEYS_TABLE;
		
		// 1. Check for duplicates
		$hash = hash( 'sha256', $key_string );
		$exists = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM {$table} WHERE key_hash = %s", $hash ) );
		
		$options = get_option( 'uwlm_options', array() );
		$allow_duplicates = $options['allow_duplicates'] ?? 0;
		
		if ( $exists && ! $allow_duplicates ) {
			return false;
		}

		// 2. Encrypt key
		$encrypted_data = UWLM_Helpers::encrypt_key( $key_string );
		if ( ! $encrypted_data ) {
			return false; // Encryption failed
		}

		// 3. Prepare data for DB
		$db_data = array(
			'license_key'     => $encrypted_data['key'],
			'iv'              => $encrypted_data['iv'],
			'key_hash'        => $hash,
			'order_id'        => $order_id,
			'product_id'      => $product_id,
			'variation_id'    => $variation_id,
			'customer_id'     => $customer_id,
			'activations'     => $activations,
			'max_activations' => $max_activations,
			'valid_for'       => $validity_days,
			'created_at'      => current_time( 'mysql' ),
			'expires_at'      => $expires_at ? date( 'Y-m-d H:i:s', strtotime( $expires_at ) ) : null,
			'status'          => $status,
		);
		
		// 4. Insert into DB
		return $wpdb->insert( $table, $db_data );
	}
	
	/**
	 * Check for and handle import/export form submissions.
	 */
	public static function handle_import_export_actions() {
		// This function is now hooked to admin_init, so we check for our specific page and action
		if ( ! isset( $_POST['action'] ) || 
			( $_POST['action'] !== 'uwlm_export_keys' && $_POST['action'] !== 'uwlm_import_keys' ) ) {
			return;
		}
		
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}

		// Handle Export
		if ( $_POST['action'] === 'uwlm_export_keys' ) {
			if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'uwlm_export_keys' ) ) {
				wp_die( 'Nonce verification failed.' );
			}
			self::process_export();
		}

		// Handle Import
		if ( $_POST['action'] === 'uwlm_import_keys' ) {
			if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'uwlm_import_keys' ) ) {
				wp_die( 'Nonce verification failed.' );
			}
			if ( ! empty( $_FILES['uwlm_import_file'] ) ) {
				self::process_import( $_FILES['uwlm_import_file'] );
			} else {
				add_action( 'admin_notices', function() {
					echo '<div class="notice notice-error"><p>' . esc_html__( 'No file was uploaded.', 'uwlm' ) . '</p></div>';
				});
			}
		}
	}
	
	/**
	 * Process the export of license keys to a CSV file.
	 */
	private static function process_export() {
		global $wpdb;
		$table = $wpdb->prefix . UWLM_LICENSE_KEYS_TABLE;
		$keys = $wpdb->get_results( "SELECT * FROM {$table} ORDER BY id ASC" );

		$filename = 'uwlm-keys-export-' . date( 'Y-m-d' ) . '.csv';

		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename=' . $filename );

		$output = fopen( 'php://output', 'w' );

		// Add header row
		fputcsv( $output, array(
			'id',
			'license_key', // Plaintext
			'product_id',
			'variation_id',
			'order_id',
			'customer_id',
			'activations',
			'max_activations',
			'status',
			'expires_at',
			'created_at',
		) );

		foreach ( $keys as $key ) {
			$plaintext_key = UWLM_Helpers::decrypt_key( $key->license_key, $key->iv );
			
			fputcsv( $output, array(
				$key->id,
				$plaintext_key ? $plaintext_key : '(Decryption Error)',
				$key->product_id,
				$key->variation_id,
				$key->order_id,
				$key->customer_id,
				$key->activations,
				$key->max_activations,
				$key->status,
				$key->expires_at,
				$key->created_at,
			) );
		}

		fclose( $output );
		exit;
	}
	
	/**
	 * Process the import of license keys from a CSV file.
	 */
	private static function process_import( $file ) {
		if ( $file['error'] !== UPLOAD_ERR_OK ) {
			add_action( 'admin_notices', function() {
				echo '<div class="notice notice-error"><p>' . esc_html__( 'File upload error.', 'uwlm' ) . '</p></div>';
			});
			return;
		}

		$file_path = $file['tmp_name'];
		
		// Check file type
		$mime_type = mime_content_type( $file_path );
		if ( ! in_array( $mime_type, array( 'text/csv', 'text/plain', 'application/csv' ) ) ) {
			add_action( 'admin_notices', function() {
				echo '<div class="notice notice-error"><p>' . esc_html__( 'Invalid file type. Please upload a .csv file.', 'uwlm' ) . '</p></div>';
			});
			return;
		}

		$handle = fopen( $file_path, 'r' );
		if ( $handle === false ) {
			add_action( 'admin_notices', function() {
				echo '<div class="notice notice-error"><p>' . esc_html__( 'Could not open the file.', 'uwlm' ) . '</p></div>';
			});
			return;
		}

		$headers = fgetcsv( $handle );
		$headers = array_map( 'trim', $headers ); // Clean headers

		// Find required column
		$key_col = array_search( 'license_key', $headers );
		$prod_col = array_search( 'product_id', $headers );

		if ( $key_col === false || $prod_col === false ) {
			add_action( 'admin_notices', function() {
				echo '<div class="notice notice-error"><p>' . esc_html__( 'CSV file is missing required headers: "license_key" and "product_id".', 'uwlm' ) . '</p></div>';
			});
			fclose( $handle );
			return;
		}
		
		// Find optional columns
		$var_col = array_search( 'variation_id', $headers );
		$max_col = array_search( 'max_activations', $headers );
		$exp_col = array_search( 'expires_at', $headers );
		$stat_col = array_search( 'status', $headers );
		$cust_col = array_search( 'customer_id', $headers );
		$ord_col = array_search( 'order_id', $headers );
		$act_col = array_search( 'activations', $headers );

		$success_count = 0;
		$error_count = 0;

		while ( ( $row = fgetcsv( $handle ) ) !== false ) {
			$key_string = trim( $row[$key_col] );
			$product_id = absint( $row[$prod_col] );
			
			if ( empty( $key_string ) || empty( $product_id ) ) {
				$error_count++;
				continue;
			}
			
			// Get product info
			$product = wc_get_product( $product_id );
			if ( ! $product ) {
				$error_count++;
				continue;
			}
			
			$parent_id = $product->is_type( 'variation' ) ? $product->get_parent_id() : $product_id;
			$variation_id = $product->is_type( 'variation' ) ? $product_id : ( $var_col !== false ? absint( $row[$var_col] ) : 0 );

			// Get optional data
			$max_activations = ( $max_col !== false && $row[$max_col] !== '' ) ? absint( $row[$max_col] ) : 1;
			$expires_at = ( $exp_col !== false && $row[$exp_col] !== '' ) ? sanitize_text_field( $row[$exp_col] ) : null;
			$status = ( $stat_col !== false && $row[$stat_col] !== '' ) ? sanitize_text_field( $row[$stat_col] ) : 'active';
			$customer_id = ( $cust_col !== false && $row[$cust_col] !== '' ) ? absint( $row[$cust_col] ) : 0;
			$order_id = ( $ord_col !== false && $row[$ord_col] !== '' ) ? absint( $row[$ord_col] ) : 0;
			$activations = ( $act_col !== false && $row[$act_col] !== '' ) ? absint( $row[$act_col] ) : 0;
			
			$validity_days = 0;
			if ( $expires_at ) {
				$today = new DateTime();
				$expiry = new DateTime( $expires_at );
				$diff = $today->diff( $expiry );
				$validity_days = $diff->days > 0 ? $diff->days : 0;
			}

			$result = self::insert_manual_key(
				$key_string, $parent_id, $variation_id, $max_activations, $validity_days, $expires_at, $status, $customer_id, $order_id, $activations
			);

			if ( $result ) {
				$success_count++;
			} else {
				$error_count++;
			}
		}
		
		fclose( $handle );

		add_action( 'admin_notices', function() use ( $success_count, $error_count ) {
			$message = sprintf( esc_html__( 'Import complete. %d keys imported successfully.', 'uwlm' ), $success_count );
			if ( $error_count > 0 ) {
				$message .= ' ' . sprintf( esc_html__( '%d keys failed to import (e.g., duplicates or missing data).', 'uwlm' ), $error_count );
			}
			echo '<div class="notice notice-success is-dismissible"><p>' . $message . '</p></div>';
		});
	}

	/**
	 * Handle AJAX for Pro License activation.
	 */
	public static function ajax_activate_pro_license() {
		check_ajax_referer( 'uwlm_admin_nonce', 'nonce' );
		
		$license_key = isset( $_POST['license_key'] ) ? sanitize_text_field( $_POST['license_key'] ) : '';
		if ( empty( $license_key ) ) {
			wp_send_json_error( array( 'message' => 'Please enter a license key.' ) );
		}

		$api_url = UWLM_PRO_STORE_URL . '/wp-json/uwlm/v1/activate';
		$product_id = UWLM_PRO_PRODUCT_ID;
		
		// We are activating *this plugin*, so we use *this plugin's* API to talk
		// to the store where it was purchased.
		$response = wp_remote_post( $api_url, array(
			'timeout' => 15,
			'headers' => array(
				'Content-Type' => 'application/json'
				// No API keys needed for activation, just the license key
			),
			'body' => json_encode( array(
				'license_key'  => $license_key,
				'product_id'   => $product_id,
				'instance_url' => home_url(),
			) ),
		) );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( array( 'message' => 'Connection Error: ' . $response->get_error_message() ) );
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		if ( $data && isset( $data['activated'] ) && $data['activated'] === true ) {
			// Success! Save the status.
			$options = get_option( 'uwlm_options', array() );
			$options['premium_status'] = 'active';
			$options['premium_license_key'] = $license_key;
			update_option( 'uwlm_options', $options );
			
			wp_send_json_success( array( 'message' => 'License activated successfully!' ) );
		} else {
			// Failed
			$error = $data['error'] ?? 'Invalid license key or communication error.';
			wp_send_json_error( array( 'message' => 'Activation Failed: ' . $error ) );
		}
	}

	/**
	 * Handle AJAX for Pro License deactivation.
	 */
	public static function ajax_deactivate_pro_license() {
		check_ajax_referer( 'uwlm_admin_nonce', 'nonce' );
		
		$options = get_option( 'uwlm_options', array() );
		$license_key = $options['premium_license_key'] ?? '';

		if ( empty( $license_key ) ) {
			wp_send_json_error( array( 'message' => 'No license key found to deactivate.' ) );
		}

		$api_url = UWLM_PRO_STORE_URL . '/wp-json/uwlm/v1/deactivate';
		
		// Deactivate on remote server
		wp_remote_post( $api_url, array(
			'timeout' => 15,
			'headers' => array(
				'Content-Type' => 'application/json'
			),
			'body' => json_encode( array(
				'license_key'  => $license_key,
				'instance_url' => home_url(),
			) ),
		) );
		
		// Don't check response, just deactivate locally regardless
		$options['premium_status'] = 'inactive';
		// $options['premium_license_key'] = ''; // Keep the key
		update_option( 'uwlm_options', $options );
		
		wp_send_json_success( array( 'message' => 'License deactivated.' ) );
	}
	
	/**
	 * Handle AJAX for generating a key string.
	 */
	public static function ajax_generate_key_string() {
		check_ajax_referer( 'uwlm_admin_nonce', 'nonce' );
		
		// Generate a key *without* saving it
		$key = UWLM_Helpers::generate_key_string();
		
		wp_send_json_success( array( 'key' => $key ) );
	}
	
	/**
	 * Handle row actions from the license key table.
	 */
	public static function handle_license_key_table_actions() {
		// This function is now hooked to admin_init
		
		// Check if we are on the right page
		if ( ! isset( $_GET['page'] ) || $_GET['page'] !== 'uwlm' ) {
			return;
		}
		
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}

		$action = isset( $_REQUEST['action'] ) ? sanitize_key( $_REQUEST['action'] ) : '';
		$key_id = isset( $_REQUEST['license_key'] ) ? absint( $_REQUEST['license_key'] ) : 0;

		if ( 'delete' === $action && $key_id ) {
			if ( ! isset( $_REQUEST['_wpnonce'] ) || ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'uwlm_delete_key' ) ) {
				wp_die( 'Nonce verification failed.' );
			}
			
			// Delete the key and its activations
			global $wpdb;
			$wpdb->delete( $wpdb->prefix . UWLM_LICENSE_KEYS_TABLE, array( 'id' => $key_id ), array( '%d' ) );
			$wpdb->delete( $wpdb->prefix . UWLM_ACTIVATIONS_TABLE, array( 'license_id' => $key_id ), array( '%d' ) );
			
			// Redirect back
			wp_redirect( admin_url( 'admin.php?page=uwlm&message=deleted' ) );
			exit;
		}
		
		// Handle bulk delete
		$action_bulk = isset( $_REQUEST['action'] ) ? sanitize_key( $_REQUEST['action'] ) : '';
		if ( -1 !== $action_bulk ) { // Support for top and bottom bulk action dropdowns
			$action_bulk = isset( $_REQUEST['action2'] ) ? sanitize_key( $_REQUEST['action2'] ) : $action_bulk;
		}

		if ( 'bulk-delete' === $action_bulk ) {
			if ( ! isset( $_REQUEST['_wpnonce'] ) || ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'bulk-licenses' ) ) {
				wp_die( 'Nonce verification failed.' );
			}
			
			$key_ids = isset( $_REQUEST['license_keys'] ) ? array_map( 'absint', $_REQUEST['license_keys'] ) : array();
			
			if ( ! empty( $key_ids ) ) {
				global $wpdb;
				$ids_placeholder = implode( ',', array_fill( 0, count( $key_ids ), '%d' ) );
				
				$wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->prefix}uwlm_license_keys WHERE id IN ( $ids_placeholder )", $key_ids ) );
				$wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->prefix}uwlm_license_activations WHERE license_id IN ( $ids_placeholder )", $key_ids ) );
				
				// Redirect back
				wp_redirect( admin_url( 'admin.php?page=uwlm&message=bulk_deleted' ) );
				exit;
			}
		}
	}
}