<?php
/**
 * Admin Class
 *
 * Handles admin menus and settings pages.
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class UWLM_Admin {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		
		// Enqueue admin scripts
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
		
		// Add form enctype for file uploads on product page
		add_action( 'post_edit_form_tag', array( $this, 'add_form_enctype' ) );
	}

	/**
	 * Add enctype to product edit form for file uploads.
	 */
	public function add_form_enctype( $post ) {
		if ( get_post_type( $post ) === 'product' ) {
			echo ' enctype="multipart/form-data"';
		}
	}
	
	/**
	 * Enqueue admin scripts and styles.
	 */
	public function enqueue_admin_scripts( $hook ) {
		// Only load on our plugin pages
		if ( strpos( $hook, 'uwlm' ) === false && $hook !== 'post.php' && $hook !== 'post-new.php' ) {
			return;
		}

		// Enqueue on all our pages + product pages
		wp_enqueue_style( 'uwlm-admin-css', UWLM_PLUGIN_URL . 'admin/css/uwlm-admin.css', array(), UWLM_VERSION );

		// Enqueue Select2 for product search on "Add New" page
		if ( $hook === 'toplevel_page_uwlm' || strpos( $hook, 'uwlm' ) !== false ) {
			wp_enqueue_script( 'wc-product-search' );
			wp_enqueue_style( 'select2' );
		}
		
		wp_enqueue_script( 'uwlm-admin-js', UWLM_PLUGIN_URL . 'admin/js/uwlm-admin.js', array( 'jquery', 'wc-product-search' ), UWLM_VERSION, true );
		
		wp_localize_script( 'uwlm-admin-js', 'uwlm_admin', array(
			'ajax_url' => admin_url( 'admin-ajax.php' ),
			'nonce'    => wp_create_nonce( 'uwlm_admin_nonce' ),
			'activating_text' => __( 'Activate License', 'uwlm' ),
			'deactivating_text' => __( 'Deactivate License', 'uwlm' ),
			'error_text' => __( 'An error occurred. Please try again.', 'uwlm' ),
		) );
	}

	/**
	 * Add admin menu pages.
	 */
	public function add_admin_menu() {
		add_menu_page(
			__( 'License Manager', 'uwlm' ),
			__( 'License Manager', 'uwlm' ),
			'manage_woocommerce',
			'uwlm',
			array( $this, 'render_license_keys_page' ),
			'dashicons-admin-network',
			56
		);

		add_submenu_page(
			'uwlm',
			__( 'License Keys', 'uwlm' ),
			__( 'License Keys', 'uwlm' ),
			'manage_woocommerce',
			'uwlm',
			array( $this, 'render_license_keys_page' )
		);
		
		add_submenu_page(
			'uwlm',
			__( 'Add License Key', 'uwlm' ),
			__( 'Add License Key', 'uwlm' ),
			'manage_woocommerce',
			'uwlm-add-new',
			array( $this, 'render_add_new_page' )
		);

		add_submenu_page(
			'uwlm',
			__( 'Import / Export', 'uwlm' ),
			__( 'Import / Export', 'uwlm' ),
			'manage_woocommerce',
			'uwlm-import-export',
			array( $this, 'render_import_export_page' )
		);

		add_submenu_page(
			'uwlm',
			__( 'Settings', 'uwlm' ),
			__( 'Settings', 'uwlm' ),
			'manage_woocommerce',
			'uwlm-settings',
			array( $this, 'render_settings_page' )
		);
	}

	/**
	 * Render the License Keys page (WP_List_Table).
	 */
	public function render_license_keys_page() {
		// --- FIX: REMOVED handler call ---
		// The handler is now on admin_init
		
		$license_keys_table = new UWLM_License_Keys_List_Table();
		$license_keys_table->prepare_items();
		?>
		<div class="wrap">
			<h1 class="wp-heading-inline"><?php esc_html_e( 'License Keys', 'uwlm' ); ?></h1>
			<a href="<?php echo esc_url( admin_url( 'admin.php?page=uwlm-add-new' ) ); ?>" class="page-title-action">
				<?php esc_html_e( 'Add New', 'uwlm' ); ?>
			</a>
			<hr class="wp-header-end">

			<?php
			// Show admin notices for delete/bulk delete
			if ( isset( $_GET['message'] ) && $_GET['message'] === 'deleted' ) {
				echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'License key deleted.', 'uwlm' ) . '</p></div>';
			}
			if ( isset( $_GET['message'] ) && $_GET['message'] === 'bulk_deleted' ) {
				echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'License keys deleted.', 'uwlm' ) . '</p></div>';
			}
			?>
			
			<form id="license-keys-filter" method="get">
				<input type="hidden" name="page" value="<?php echo esc_attr( $_REQUEST['page'] ); ?>" />
				<?php $license_keys_table->search_box( 'search', 'search_id' ); ?>
				<?php $license_keys_table->display(); ?>
			</form>
		</div>
		<?php
	}

	/**
	 * Render the Add New License Key page.
	 */
	public function render_add_new_page() {
		// --- FIX: REMOVED handler call ---
		// The handler is now on admin_init

		// Use wc_get_products for a more efficient query
		$all_products = wc_get_products( array( 'limit' => -1, 'status' => 'publish' ) );
		$licensed_products = array();
		
		foreach( $all_products as $product ) {
			if ( $product->get_meta('_uwlm_grants_license') === 'yes' && $product->is_type('simple') ) {
				$licensed_products[] = $product;
			} elseif ( $product->is_type('variable') ) {
				// For variables, add the parent and we will list variations
				$licensed_products[] = $product;
			}
		}
		
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Add New License Key', 'uwlm' ); ?></h1>
			<p><?php esc_html_e( 'Manually create a license key and assign it to a product.', 'uwlm' ); ?></p>
			
			<form action="" method="POST" class="uwlm-add-new-form">
				<?php wp_nonce_field( 'uwlm_add_new' ); ?>
				<input type="hidden" name="action" value="uwlm_add_new_license">
				
				<table class="form-table">
					<tbody>
						<!-- License Key -->
						<tr>
							<th scope="row">
								<label for="license_key"><?php esc_html_e( 'License Key', 'uwlm' ); ?></label>
							</th>
							<td>
								<input type="text" id="license_key" name="license_key" class="regular-text">
								<button type="button" class="button button-secondary" id="uwlm-generate-key-btn"><?php esc_html_e( 'Generate Key', 'uwlm' ); ?></button>
								<p class="description"><?php esc_html_e( 'Enter a license key or generate a random one. (e.g., XXXXX-XXXXX-XXXXX-XXXXX)', 'uwlm' ); ?></p>
							</td>
						</tr>
						
						<!-- Product -->
						<tr>
							<th scope="row">
								<label for="product_id"><?php esc_html_e( 'Product', 'uwlm' ); ?></label>
							</th>
							<td>
								<select id="product_id" name="product_id" class="wc-product-search" data-placeholder="<?php esc_attr_e( 'Search for a product...', 'uwlm' ); ?>" style="width: 350px;">
									<option value=""><?php esc_html_e( 'Select a product...', 'uwlm' ); ?></option>
									<?php
									foreach ( $licensed_products as $product ) {
										if( $product->is_type('variable') ) {
											echo '<optgroup label="' . esc_attr( $product->get_name() ) . '">';
											foreach ( $product->get_children() as $variation_id ) {
												$variation = wc_get_product( $variation_id );
												if ( ! $variation || $variation->get_meta('_uwlm_grants_license') !== 'yes' ) {
													continue;
												}
												echo '<option value="' . esc_attr( $variation_id ) . '"> &nbsp; ' . esc_html( $variation->get_formatted_name() ) . '</option>';
											}
											echo '</optgroup>';
										} else {
											// Only simple products that grant licenses
											echo '<option value="' . esc_attr( $product->get_id() ) . '">' . esc_html( $product->get_name() ) . '</option>';
										}
									}
									?>
								</select>
								<p class="description"><?php esc_html_e( 'Select the product or variation this license is for.', 'uwlm' ); ?></p>
							</td>
						</tr>
						
						<!-- Max Activations -->
						<tr>
							<th scope="row">
								<label for="max_activations"><?php esc_html_e( 'Max Activations', 'uwlm' ); ?></label>
							</th>
							<td>
								<input type="number" id="max_activations" name="max_activations" value="1" min="0" step="1" class="small-text">
								<p class="description"><?php esc_html_e( 'Number of sites this key can be activated on. 0 for unlimited.', 'uwlm' ); ?></p>
							</td>
						</tr>

						<!-- Expires At -->
						<tr>
							<th scope="row">
								<label for="expires_at"><?php esc_html_e( 'Expiration Date', 'uwlm' ); ?></label>
							</th>
							<td>
								<input type="date" id="expires_at" name="expires_at" class="regular-text" style="width: 200px;">
								<p class="description"><?php esc_html_e( 'Leave blank for a lifetime license.', 'uwlm' ); ?></p>
							</td>
						</tr>

					</tbody>
				</table>
				
				<?php submit_button( __( 'Create License Key', 'uwlm' ) ); ?>
			</form>
		</div>
		<?php
	}

	/**
	 * Render the Import / Export page.
	 */
	public function render_import_export_page() {
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Import / Export License Keys', 'uwlm' ); ?></h1>
			
			<div class="uwlm-admin-card-wrapper">
				<!-- Import -->
				<div class="uwlm-admin-card">
					<h2><?php esc_html_e( 'Import Keys from CSV', 'uwlm' ); ?></h2>
					<p><?php esc_html_e( 'Upload a CSV file to bulk-import license keys. This is useful for migrating from another system or adding pre-generated keys.', 'uwlm' ); ?></p>
					<form action="" method="POST" enctype="multipart/form-data">
						<?php wp_nonce_field( 'uwlm_import_keys' ); ?>
						<input type="hidden" name="action" value="uwlm_import_keys">
						
						<table class="form-table">
							<tbody>
								<tr>
									<th scope="row">
										<label for="uwlm_import_file"><?php esc_html_e( 'CSV File', 'uwlm' ); ?></label>
									</th>
									<td>
										<input type="file" id="uwlm_import_file" name="uwlm_import_file" accept=".csv">
										<p class="description"><?php esc_html_e( 'Your CSV must have a header row with columns like: license_key, product_id, max_activations, expires_at (YYYY-MM-DD), status', 'uwlm' ); ?></p>
									</td>
								</tr>
							</tbody>
						</table>
						<?php submit_button( __( 'Import Keys', 'uwlm' ), 'primary' ); ?>
					</form>
				</div>
				
				<!-- Export -->
				<div class="uwlm-admin-card">
					<h2><?php esc_html_e( 'Export Keys to CSV', 'uwlm' ); ?></h2>
					<p><?php esc_html_e( 'Download a CSV file of all license keys in your database. This includes decrypted keys for your records.', 'uwlm' ); ?></p>
					<form action="" method="POST">
						<?php wp_nonce_field( 'uwlm_export_keys' ); ?>
						<input type="hidden" name="action" value="uwlm_export_keys">
						<?php submit_button( __( 'Export All Keys', 'uwlm' ), 'secondary' ); ?>
					</form>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Render the Settings page.
	 */
	public function render_settings_page() {
		$active_tab = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'general';
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'License Manager Settings', 'uwlm' ); ?></h1>
			
			<h2 class="nav-tab-wrapper">
				<a href="?page=uwlm-settings&tab=general" class="nav-tab <?php echo $active_tab === 'general' ? 'nav-tab-active' : ''; ?>">
					<?php esc_html_e( 'General', 'uwlm' ); ?>
				</a>
				<a href="?page=uwlm-settings&tab=api" class="nav-tab <?php echo $active_tab === 'api' ? 'nav-tab-active' : ''; ?>">
					<?php esc_html_e( 'API Settings', 'uwlm' ); ?>
				</a>
				<a href="?page=uwlm-settings&tab=pro" class="nav-tab <?php echo $active_tab === 'pro' ? 'nav-tab-active' : ''; ?>">
					<?php esc_html_e( 'Upgrade to Pro', 'uwlm' ); ?>
				</a>
			</h2>

			<form action="options.php" method="post">
				<?php
				settings_fields( 'uwlm_settings' );
				
				if ( $active_tab === 'general' ) {
					do_settings_sections( 'uwlm_settings_general' );
				} elseif ( $active_tab === 'api' ) {
					do_settings_sections( 'uwlm_settings_api' );
				} elseif ( $active_tab === 'pro' ) {
					do_settings_sections( 'uwlm_settings_pro' );
				}
				
				// Only show save button on general and api tabs
				if ( $active_tab !== 'pro' ) {
					submit_button();
				}
				?>
			</form>
		</div>
		<?php
	}

	/**
	 * Register plugin settings.
	 */
	public function register_settings() {
		register_setting( 'uwlm_settings', 'uwlm_options', array( $this, 'sanitize_settings' ) );

		// General Section
		add_settings_section(
			'uwlm_general',
			__( 'General Settings', 'uwlm' ),
			'__return_false',
			'uwlm_settings_general'
		);

		// Order Status
		add_settings_field(
			'uwlm_order_status',
			__( 'Trigger License Generation', 'uwlm' ),
			array( $this, 'render_order_status_field' ),
			'uwlm_settings_general',
			'uwlm_general'
		);
		
		// Allow Duplicates
		add_settings_field(
			'uwlm_allow_duplicates',
			__( 'Allow Duplicate Keys', 'uwlm' ),
			array( $this, 'render_allow_duplicates_field' ),
			'uwlm_settings_general',
			'uwlm_general'
		);

		// API Section
		add_settings_section(
			'uwlm_api',
			__( 'API Settings', 'uwlm' ),
			array( $this, 'render_api_section_text' ),
			'uwlm_settings_api'
		);

		// API Key
		add_settings_field(
			'uwlm_api_key',
			__( 'API Public Key', 'uwlm' ),
			array( $this, 'render_api_key_field' ),
			'uwlm_settings_api',
			'uwlm_api'
		);

		// API Secret
		add_settings_field(
			'uwlm_api_secret',
			__( 'API Secret Key', 'uwlm' ),
			array( $this, 'render_api_secret_field' ),
			'uwlm_settings_api',
			'uwlm_api'
		);
		
		// --- NEW: Pro Section ---
		add_settings_section(
			'uwlm_pro',
			__( 'Pro License Activation', 'uwlm' ),
			array( $this, 'render_pro_section_text' ),
			'uwlm_settings_pro'
		);
		
		// Pro License Key
		add_settings_field(
			'uwlm_premium_license_key',
			__( 'Pro License Key', 'uwlm' ),
			array( $this, 'render_pro_license_key_field' ),
			'uwlm_settings_pro',
			'uwlm_pro'
		);
	}

	/**
	 * Sanitize settings.
	 *
	 * @param array $input The settings to sanitize.
	 * @return array Sanitized settings.
	 */
	public function sanitize_settings( $input ) {
		$output = get_option( 'uwlm_options', array() );

		$output['order_status']      = isset( $input['order_status'] ) ? sanitize_text_field( $input['order_status'] ) : 'wc-completed';
		$output['api_key']           = isset( $input['api_key'] ) ? sanitize_text_field( $input['api_key'] ) : '';
		$output['api_secret']        = isset( $input['api_secret'] ) ? sanitize_text_field( $input['api_secret'] ) : '';
		$output['allow_duplicates']  = isset( $input['allow_duplicates'] ) ? 1 : 0;
		$output['premium_license_key'] = isset( $input['premium_license_key'] ) ? sanitize_text_field( $input['premium_license_key'] ) : '';
		
		// 'premium_status' is not saved here, it's saved via AJAX
		$output['premium_status']    = $output['premium_status'] ?? 'inactive';
		
		// Regenerate API keys if buttons were clicked
		if ( isset( $_POST['regenerate_api_key'] ) ) { // Check POST data, not $input
			$output['api_key'] = 'pub_' . bin2hex( random_bytes( 16 ) );
		}
		if ( isset( $_POST['regenerate_api_secret'] ) ) { // Check POST data, not $input
			$output['api_secret'] = 'sec_' . bin2hex( random_bytes( 32 ) );
		}

		return $output;
	}

	/**
	 * Render settings fields.
	 */
	public function render_order_status_field() {
		$options      = get_option( 'uwlm_options', array() );
		$order_status = $options['order_status'] ?? 'wc-completed';
		$statuses     = wc_get_order_statuses();
		?>
		<select id="uwlm_order_status" name="uwlm_options[order_status]">
			<?php foreach ( $statuses as $key => $label ) : ?>
				<option value="<?php echo esc_attr( $key ); ?>" <?php selected( $order_status, $key ); ?>><?php echo esc_html( $label ); ?></option>
			<?php endforeach; ?>
		</select>
		<p class="description"><?php esc_html_e( 'The order status at which license keys should be generated and delivered.', 'uwlm' ); ?></p>
		<?php
	}

	public function render_api_section_text() {
		echo '<p>' . esc_html__( 'These keys are used to authenticate with the REST API.', 'uwlm' ) . '</p>';
		echo '<p><strong>' . esc_html__( 'Secret Key Warning:', 'uwlm' ) . '</strong> ' . esc_html__( 'Your API Secret Key is private. Do not share it. It is only required for server-to-server calls.', 'uwlm' ) . '</p>';
	}

	public function render_api_key_field() {
		$options = get_option( 'uwlm_options', array() );
		$key     = $options['api_key'] ?? '';
		?>
		<input type="text" id="uwlm_api_key" name="uwlm_options[api_key]" value="<?php echo esc_attr( $key ); ?>" class="regular-text" readonly />
		<button type="submit" name="regenerate_api_key" class="button"><?php esc_html_e( 'Regenerate', 'uwlm' ); ?></button>
		<?php
	}

	public function render_api_secret_field() {
		$options = get_option( 'uwlm_options', array() );
		$secret  = $options['api_secret'] ?? '';
		?>
		<input type="text" id="uwlm_api_secret" name="uwlm_options[api_secret]" value="<?php echo esc_attr( $secret ); ?>" class="regular-text" readonly />
		<button type="submit" name="regenerate_api_secret" class="button"><?php esc_html_e( 'Regenerate', 'uwlm' ); ?></button>
		<?php
	}

	public function render_allow_duplicates_field() {
		$options = get_option( 'uwlm_options', array() );
		$allowed = $options['allow_duplicates'] ?? 0;
		?>
		<label>
			<input type="checkbox" id="uwlm_allow_duplicates" name="uwlm_options[allow_duplicates]" value="1" <?php checked( $allowed, 1 ); ?> />
			<?php esc_html_e( 'Allow duplicate license keys to be imported.', 'uwlm' ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'Not recommended. Auto-generated keys are always unique.', 'uwlm' ); ?></p>
		<?php
	}

	/**
	 * --- NEW ---
	 * Render Pro section fields.
	 */
	public function render_pro_section_text() {
		$pro_url = 'https://www.ultimatemultimediaconsult.com/product/ultimate-woocommerce-license-manager/';
		echo '<p>' . sprintf( wp_kses_post( __( 'Thank you for using Ultimate WooCommerce Licenses Manager! The free version is limited to 1 licensed product. <br>To unlock unlimited products, please <a href="%s" target="_blank">purchase a Pro license key</a>.', 'uwlm' ) ), esc_url( $pro_url ) ) . '</p>';
	}

	public function render_pro_license_key_field() {
		$options      = get_option( 'uwlm_options', array() );
		$key          = $options['premium_license_key'] ?? '';
		$status       = $options['premium_status'] ?? 'inactive';
		$status_text  = $status === 'active' ? __( 'ACTIVE', 'uwlm' ) : __( 'INACTIVE', 'uwlm' );
		$status_class = $status === 'active' ? 'uwlm-status-active' : 'uwlm-status-inactive';
		?>
		<input type="text" id="uwlm_premium_license_key" name="uwlm_options[premium_license_key]" value="<?php echo esc_attr( $key ); ?>" class="regular-text" <?php echo $status === 'active' ? 'readonly' : ''; ?>>
		
		<?php if ( $status === 'active' ) : ?>
			<button type="button" class="button button-secondary" id="uwlm-pro-license-btn" data-action="deactivate">
				<?php esc_html_e( 'Deactivate License', 'uwlm' ); ?>
			</button>
		<?php else : ?>
			<button type="button" class="button button-primary" id="uwlm-pro-license-btn" data-action="activate">
				<?php esc_html_e( 'Activate License', 'uwlm' ); ?>
			</button>
		<?php endif; ?>
		
		<p class="description">
			<?php esc_html_e( 'Your license status is:', 'uwlm' ); ?>
			<strong id="uwlm-pro-status" class="<?php echo esc_attr( $status_class ); ?>" style="margin-left: 5px; padding: 3px 6px; border-radius: 3px;">
				<?php echo esc_html( $status_text ); ?>
			</strong>
		</p>
		<div id="uwlm-pro-message" style="margin-top: 10px;"></div>
		<?php
	}
}