<?php
/**
 * License Keys List Table Class
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP_List_Table is not loaded automatically
if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class UWLM_License_Keys_List_Table extends WP_List_Table {

	/**
	 * Constructor.
	 */
	public function __construct() {
		parent::__construct(
			array(
				'singular' => __( 'License Key', 'uwlm' ),
				'plural'   => __( 'License Keys', 'uwlm' ),
				'ajax'     => false,
			)
		);
	}

	/**
	 * Get columns.
	 *
	 * @return array
	 */
	public function get_columns() {
		return array(
			'cb'          => '<input type="checkbox" />',
			'license_key' => __( 'License Key', 'uwlm' ),
			'order'       => __( 'Order', 'uwlm' ),
			'product'     => __( 'Product', 'uwlm' ),
			'customer'    => __( 'Customer', 'uwlm' ),
			'activations' => __( 'Activations', 'uwlm' ),
			'created_at'  => __( 'Created', 'uwlm' ),
			'expires_at'  => __( 'Expires', 'uwlm' ),
			'status'      => __( 'Status', 'uwlm' ),
		);
	}
	
	/**
	 * Hidden columns.
	 */
	public function get_hidden_columns() {
		return array( 'id' ); // Add 'id' to hidden columns
	}

	/**
	 * Get sortable columns.
	 *
	 * @return array
	 */
	public function get_sortable_columns() {
		return array(
			'order'      => array( 'order_id', false ),
			'product'    => array( 'product_id', false ),
			'customer'   => array( 'customer_id', false ),
			'created_at' => array( 'created_at', true ), // True for default order
			'expires_at' => array( 'expires_at', false ),
			'status'     => array( 'status', false ),
		);
	}

	/**
	 * Get data for the table.
	 *
	 * @return array
	 */
	private function get_table_data() {
		global $wpdb;
		$table = $wpdb->prefix . UWLM_LICENSE_KEYS_TABLE;

		// Sorting
		$orderby = isset( $_GET['orderby'] ) ? sanitize_key( $_GET['orderby'] ) : 'created_at';
		$order   = isset( $_GET['order'] ) ? sanitize_key( $_GET['order'] ) : 'DESC';

		// Pagination
		$per_page     = $this->get_items_per_page( 'license_keys_per_page', 20 );
		$current_page = $this->get_pagenum();
		$offset       = ( $current_page - 1 ) * $per_page;

		// Query
		$sql = $wpdb->prepare(
			"SELECT * FROM {$table} ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d",
			$per_page,
			$offset
		);
		
		$data = $wpdb->get_results( $sql, ARRAY_A );

		return $data;
	}

	/**
	 * Get total number of items.
	 *
	 * @return int
	 */
	private function get_total_items() {
		global $wpdb;
		$table = $wpdb->prefix . UWLM_LICENSE_KEYS_TABLE;
		return $wpdb->get_var( "SELECT COUNT(id) FROM {$table}" );
	}

	/**
	 * Prepare items for display.
	 */
	public function prepare_items() {
		$this->_column_headers = array( $this->get_columns(), $this->get_hidden_columns(), $this->get_sortable_columns() );
		$this->items           = $this->get_table_data();

		$this->set_pagination_args(
			array(
				'total_items' => $this->get_total_items(),
				'per_page'    => $this->get_items_per_page( 'license_keys_per_page', 20 ),
			)
		);
	}
	
	/**
	 * Handle row display.
	 */
	public function single_row( $item ) {
		echo '<tr data-license-id="' . esc_attr( $item['id'] ) . '">';
		$this->single_row_columns( $item );
		echo '</tr>';
		
		// --- NEW: Add hidden row for activations ---
		$activations = UWLM_DB::get_activations( $item['id'] );
		?>
		<tr class="uwlm-activations-row" id="uwlm-activations-for-<?php echo esc_attr( $item['id'] ); ?>" style="display: none;">
			<td colspan="<?php echo count( $this->get_columns() ); ?>">
				<div class="uwlm-activations-container">
					<h4><?php esc_html_e( 'Active Instances', 'uwlm' ); ?> (<?php echo count( $activations ); ?>)</h4>
					<?php if ( empty( $activations ) ) : ?>
						<p><?php esc_html_e( 'No active instances.', 'uwlm' ); ?></p>
					<?php else : ?>
						<ul class="uwlm-activations-list">
							<?php foreach ( $activations as $activation ) : ?>
								<li>
									<span><?php echo esc_html( $activation->instance_url ); ?></span>
									<small><?php echo esc_html( date_i18n( get_option( 'date_format' ), strtotime( $activation->activated_at ) ) ); ?></small>
								</li>
							<?php endforeach; ?>
						</ul>
					<?php endif; ?>
				</div>
			</td>
		</tr>
		<?php
	}

	/**
	 * Render default column.
	 *
	 * @param array  $item
	 * @param string $column_name
	 * @return mixed
	 */
	public function column_default( $item, $column_name ) {
		switch ( $column_name ) {
			case 'id': // Return ID for hidden column
				return $item['id'];
			case 'created_at':
				return date_i18n( get_option( 'date_format' ), strtotime( $item['created_at'] ) );
			case 'expires_at':
				return $item['expires_at'] ? date_i18n( get_option( 'date_format' ), strtotime( $item['expires_at'] ) ) : __( 'Lifetime', 'uwlm' );
			case 'status':
				$status_text = ucfirst( $item['status'] );
				if ( $item['status'] === 'active' && $item['expires_at'] && strtotime( $item['expires_at'] ) < time() ) {
					$status_text = __( 'Expired', 'uwlm' );
					// Update status in DB if needed (can be intensive)
				}
				return '<span class="uwlm-status uwlm-status-' . esc_attr( $item['status'] ) . '">' . esc_html( $status_text ) . '</span>';
			default:
				return $item[ $column_name ] ? esc_html( $item[ $column_name ] ) : '—';
		}
	}

	/**
	 * Render checkbox column.
	 *
	 * @param array $item
	 * @return string
	 */
	public function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" name="license_key[]" value="%s" />',
			$item['id']
		);
	}

	/**
	 * Render License Key column.
	 *
	 * @param array $item
	 * @return string
	 */
	public FUnction column_license_key( $item ) {
		// Decrypt to show partial key
		$plaintext_key = UWLM_Helpers::decrypt_key( $item['license_key'], $item['iv'] );
		$display_key   = $plaintext_key ? ( '****-****-****-' . substr( $plaintext_key, -5 ) ) : __( 'Error decrypting', 'uwlm' );

		// --- UPDATED ACTIONS ---
		$delete_nonce = wp_create_nonce( 'uwlm_delete_key' );
		$actions = array(
			//'edit'   => sprintf( '<a href="?page=%s&action=%s&license_key=%s">Edit</a>', 'uwlm-edit-key', 'edit', $item['id'] ), // TODO: Build edit page
			'delete' => sprintf(
				'<a href="?page=%s&action=%s&license_key=%s&_wpnonce=%s" onclick="return confirm(\'Are you sure you want to permanently delete this key?\')">Delete</a>',
				esc_attr( $_REQUEST['page'] ), 'delete', $item['id'], $delete_nonce
			),
		);
		return '<strong>' . esc_html( $display_key ) . '</strong>' . $this->row_actions( $actions );
	}

	/**
	 * Render Order column.
	 *
	 * @param array $item
	 * @return string
	 */
	public function column_order( $item ) {
		if ( empty( $item['order_id'] ) ) {
			return '<em>(Manually Added)</em>';
		}
		$order = wc_get_order( $item['order_id'] );
		if ( ! $order ) {
			return '—';
		}
		$order_url = $order->get_edit_order_url();
		return $order_url ? sprintf( '<a href="%s">#%s</a>', esc_url( $order_url ), esc_html( $item['order_id'] ) ) : '—';
	}

	/**
	 * Render Product column.
	 *
	 * @param array $item
	 * @return string
	 */
	public function column_product( $item ) {
		$product_id = $item['variation_id'] ? $item['variation_id'] : $item['product_id'];
		$product = wc_get_product( $product_id );
		
		if ( ! $product ) {
			return get_the_title( $item['product_id'] ); // Fallback
		}
		
		$product_url = get_edit_post_link( $product_id );
		return $product_url ? sprintf( '<a href="%s">%s</a>', esc_url( $product_url ), $product->get_name() ) : $product->get_name();
	}

	/**
	 * Render Customer column.
	 *
	 * @param array $item
	 * @return string
	 */
	public function column_customer( $item ) {
		if ( empty( $item['customer_id'] ) ) {
			return '—';
		}
		$user = get_user_by( 'id', $item['customer_id'] );
		if ( ! $user ) {
			return '—';
		}
		$user_url = get_edit_user_link( $user->ID );
		return $user_url ? sprintf( '<a href="%s">%s</a>', esc_url( $user_url ), esc_html( $user->display_name ) ) : esc_html( $user->display_name );
	}

	/**
	 * Render Activations column.
	 *
	 * @param array $item
	 * @return string
	 */
	public function column_activations( $item ) {
		$max = $item['max_activations'] > 0 ? $item['max_activations'] : '∞';
		$count = $item['activations'];
		
		$html = sprintf( '%d / %s', $count, $max );
		
		if ( $count > 0 ) {
			$html .= ' <a href="#" class="uwlm-view-activations" data-id="' . $item['id'] . '">' . __( 'View', 'uwlm' ) . '</a>';
		}
		
		return $html;
	}
}