<?php
/**
 * Handles the "License Manager" tab in the WooCommerce product data meta box.
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class UWLM_Product_Settings {

	/**
	 * Constructor.
	 */
	public function __construct() {
		// Add tab to product page
		add_filter( 'woocommerce_product_data_tabs', array( $this, 'add_license_product_tab' ) );
		
		// Add panel for the tab
		add_action( 'woocommerce_product_data_panels', array( $this, 'render_product_data_panel' ) );
		
		// Save simple product meta
		add_action( 'woocommerce_process_product_meta', array( $this, 'save_product_meta' ), 20 );
		
		// Add variation settings
		add_action( 'woocommerce_product_after_variable_attributes', array( $this, 'render_variation_settings' ), 10, 3 );
		
		// Save variation meta
		add_action( 'woocommerce_save_product_variation', array( $this, 'save_variation_meta' ), 10, 2 );
	}

	/**
	 * Add the "License Manager" tab.
	 *
	 * @param array $tabs The existing product data tabs.
	 * @return array The modified tabs.
	 */
	public function add_license_product_tab( $tabs ) {
		$tabs['uwlm_license'] = array(
			'label'    => __( 'License Manager', 'uwlm' ),
			'target'   => 'uwlm_license_data',
			'class'    => array( 'show_if_simple', 'show_if_variable' ),
			'priority' => 80,
		);
		return $tabs;
	}

	/**
	 * Render the HTML for the product data panel.
	 */
	public function render_product_data_panel() {
		
		// --- THIS IS THE FIX ---
		// Manually include the file that defines woocommerce_wp_textarea() if it doesn't exist.
		// This permanently solves the fatal error.
		if ( ! function_exists( 'woocommerce_wp_textarea' ) ) {
			if ( defined( 'WC_PLUGIN_PATH' ) ) {
				include_once( WC_PLUGIN_PATH . 'includes/admin/wc-meta-box-functions.php' );
			} else {
				// Fallback if constant is not defined (very unlikely)
				include_once( WP_PLUGIN_DIR . '/woocommerce/includes/admin/wc-meta-box-functions.php' );
			}
		}
		// --- END FIX ---
		
		global $post, $thepostid;
		
		// Use $thepostid on 'Add New' screen, $post->ID on 'Edit' screen
		$product_id = $thepostid ? $thepostid : $post->ID;
		$product = wc_get_product( $product_id );

		// Get saved values, or provide defaults
		$grants_license    = $product_id ? $product->get_meta( '_uwlm_grants_license' ) : 'no';
		$max_activations = $product_id ? $product->get_meta( '_uwlm_max_activations' ) : 1;
		$validity_days   = $product_id ? $product->get_meta( '_uwlm_validity_days' ) : 365;
		$auto_generate   = $product_id ? $product->get_meta( '_uwlm_auto_generate' ) : 'yes';

		// Get file upload data (with defaults to prevent errors on 'Add New')
		$current_version = $product_id ? $product->get_meta( '_uwlm_current_version' ) : '';
		$changelog       = $product_id ? $product->get_meta( '_uwlm_changelog' ) : '';
		$file_path       = $product_id ? $product->get_meta( '_uwlm_file_path' ) : '';
		$file_name       = $file_path ? basename( $file_path ) : '';
		
		// Freemium restrictions
		$is_premium = uwlm_is_premium();
		$disabled_attr = '';
		$upgrade_message = '';
		
		if ( ! $is_premium ) {
			global $wpdb;
			// Count distinct products that grant licenses
			$count = $wpdb->get_var(
				"SELECT COUNT(DISTINCT post_id)
				FROM $wpdb->postmeta
				WHERE meta_key = '_uwlm_grants_license' AND meta_value = 'yes'"
			);
			
			// If 1 or more products are *already* licensed, and this *isn't* one of them...
			if ( (int) $count >= 1 && $grants_license !== 'yes' ) {
				$disabled_attr = 'disabled="disabled"';
				$upgrade_message = '<p class="description uwlm-pro-notice">' .
								sprintf( 
									wp_kses_post( __( 'The free version is limited to 1 licensed product. To license unlimited products, please <a href="%s" target="_blank">Upgrade to Pro</a>.', 'uwlm' ) ),
									admin_url( 'admin.php?page=uwlm-settings&tab=pro' )
								) .
								'</p>';
			}
		}

		?>
		<div id="uwlm_license_data" class="panel woocommerce_options_panel">
			
			<?php echo wp_kses_post( $upgrade_message ); ?>

			<div class="options_group">
				<?php
				// Checkbox: This product grants a license
				woocommerce_wp_checkbox( array(
					'id'          => '_uwlm_grants_license',
					'label'       => __( 'This product grants a license', 'uwlm' ),
					'description' => __( 'Enable this to activate the License Manager options for this product.', 'uwlm' ),
					'value'       => $grants_license,
					'cbvalue'     => 'yes',
					'disabled'    => $disabled_attr,
				) );

				// Number: Max Activations
				woocommerce_wp_text_input( array(
					'id'          => '_uwlm_max_activations',
					'label'       => __( 'Max Activations', 'uwlm' ),
					'description' => __( 'Number of sites this key can be activated on. 0 for unlimited.', 'uwlm' ),
					'type'        => 'number',
					'value'       => $max_activations,
					'custom_attributes' => array( 'min' => '0', 'step' => '1' ),
					'disabled'    => $disabled_attr,
				) );

				// Number: Validity (days)
				woocommerce_wp_text_input( array(
					'id'          => '_uwlm_validity_days',
					'label'       => __( 'Validity (days)', 'uwlm' ),
					'description' => __( 'Number of days the license is valid for after purchase. 0 for lifetime.', 'uwlm' ),
					'type'        => 'number',
					'value'       => $validity_days,
					'custom_attributes' => array( 'min' => '0', 'step' => '1' ),
					'disabled'    => $disabled_attr,
				) );
				
				// Checkbox: Auto-generate keys
				woocommerce_wp_checkbox( array(
					'id'          => '_uwlm_auto_generate',
					'label'       => __( 'Auto-generate Keys', 'uwlm' ),
					'description' => __( 'Automatically generate a new license key when this product is purchased.', 'uwlm' ),
					'value'       => $auto_generate,
					'cbvalue'     => 'yes',
					'disabled'    => $disabled_attr,
				) );

				// TODO: Add support for "Import License Keys" textarea
				?>
			</div>
			
			<div class="options_group">
				<h4 style="margin-bottom: 10px;"><?php esc_html_e( 'Software File & Version Control', 'uwlm' ); ?></h4>
				<?php
				
				// Text: Current Version
				woocommerce_wp_text_input( array(
					'id'          => '_uwlm_current_version',
					'label'       => __( 'Current Version', 'uwlm' ),
					'description' => __( 'e.g., 1.0.0. This is sent to the client for update checks.', 'uwlm' ),
					'type'        => 'text',
					'value'       => $current_version,
					'disabled'    => $disabled_attr,
				) );
				
				// Textarea: Changelog
				woocommerce_wp_textarea_input( array(
					'id'          => '_uwlm_changelog',
					'label'       => __( 'Changelog', 'uwlm' ),
					'description' => __( 'Enter the changelog for the current version.', 'uwlm' ),
					'value'       => $changelog,
					'style'       => 'height: 100px;',
					'disabled'    => $disabled_attr,
				) );
				
				// File Upload
				?>
				<p class="form-field _uwlm_file_path_field">
					<label for="_uwlm_file_path"><?php esc_html_e( 'Product File', 'uwlm' ); ?></label>
					<span class="file-upload-wrapper">
						<input type="file" id="_uwlm_file_path" name="_uwlm_file_path" accept=".zip" <?php echo esc_attr( $disabled_attr ); ?>>
						<span class="description">
							<?php
							if ( $file_name ) {
								echo sprintf( esc_html__( 'Current file: %s. Upload a new .zip file to replace it.', 'uwlm' ), '<strong>' . esc_html( $file_name ) . '</strong>' );
							} else {
								esc_html_e( 'Upload the software .zip file.', 'uwlm' );
							}
							?>
						</span>
					</span>
				</p>
			</div>

		</div>
		<?php
	}
	
	/**
	 * Render the settings for variations.
	 */
	public function render_variation_settings( $loop, $variation_data, $variation ) {
		// Get variation product object
		$variation_product = wc_get_product( $variation->ID );
		if ( ! $variation_product ) {
			return;
		}

		// Get saved values
		$grants_license    = $variation_product->get_meta( '_uwlm_grants_license' ) ?: 'no';
		$max_activations = $variation_product->get_meta( '_uwlm_max_activations' ) ?: 1;
		$validity_days   = $variation_product->get_meta( '_uwlm_validity_days' ) ?: 365;
		$auto_generate   = $variation_product->get_meta( '_uwlm_auto_generate' ) ?: 'yes';
		
		// Freemium restrictions
		$is_premium = uwlm_is_premium();
		$disabled_attr = '';
		$upgrade_message = '';
		
		if ( ! $is_premium ) {
			global $wpdb;
			$count = $wpdb->get_var(
				"SELECT COUNT(DISTINCT post_id)
				FROM $wpdb->postmeta
				WHERE meta_key = '_uwlm_grants_license' AND meta_value = 'yes'"
			);
			
			// If 1 or more products are *already* licensed...
			if ( (int) $count >= 1 ) {
				// We need to check if EITHER the parent OR this variation is the one that's licensed
				$parent_grants_license = get_post_meta( $variation->post_parent, '_uwlm_grants_license', true ) === 'yes';
				
				if ( $grants_license !== 'yes' && ! $parent_grants_license ) {
					$disabled_attr = 'disabled="disabled"';
					$upgrade_message = '<p class="description uwlm-pro-notice" style="grid-column: span 2;">' .
									sprintf( 
										wp_kses_post( __( 'Free version limited to 1 product. <a href="%s" target="_blank">Upgrade to Pro</a>.', 'uwlm' ) ),
										admin_url( 'admin.php?page=uwlm-settings&tab=pro' )
									) .
									'</p>';
				}
			}
		}

		?>
		<div class="options_group uwlm-variation-settings">
			<h4 style="grid-column: span 2; margin: 0; padding-top: 10px; border-top: 1px solid #eee;"><?php esc_html_e( 'License Manager', 'uwlm' ); ?></h4>
			
			<?php echo wp_kses_post( $upgrade_message ); ?>
			
			<?php
			// Checkbox: This product grants a license
			woocommerce_wp_checkbox( array(
				'id'            => "_uwlm_grants_license[{$loop}]",
				'label'         => __( 'This variation grants a license', 'uwlm' ),
				'description'   => '',
				'value'         => $grants_license,
				'cbvalue'       => 'yes',
				'wrapper_class' => 'form-row form-row-full',
				'disabled'      => $disabled_attr,
			) );

			// Number: Max Activations
			woocommerce_wp_text_input( array(
				'id'            => "_uwlm_max_activations[{$loop}]",
				'label'         => __( 'Max Activations', 'uwlm' ),
				'type'          => 'number',
				'value'         => $max_activations,
				'custom_attributes' => array( 'min' => '0', 'step' => '1' ),
				'wrapper_class' => 'form-row form-row-first',
				'disabled'      => $disabled_attr,
			) );

			// Number: Validity (days)
			woocommerce_wp_text_input( array(
				'id'            => "_uwlm_validity_days[{$loop}]",
				'label'         => __( 'Validity (days)', 'uwlm' ),
				'type'          => 'number',
				'value'         => $validity_days,
				'custom_attributes' => array( 'min' => '0', 'step' => '1' ),
				'wrapper_class' => 'form-row form-row-last',
				'disabled'      => $disabled_attr,
			) );
			
			// Checkbox: Auto-generate keys
			woocommerce_wp_checkbox( array(
				'id'            => "_uwlm_auto_generate[{$loop}]",
				'label'         => __( 'Auto-generate Keys', 'uwlm' ),
				'description'   => '',
				'value'         => $auto_generate,
				'cbvalue'       => 'yes',
				'wrapper_class' => 'form-row form-row-full',
				'disabled'      => $disabled_attr,
			) );
			?>
		</div>
		<?php
	}

	/**
	 * Save the product meta.
	 */
	public function save_product_meta( $post_id ) {
		$product = wc_get_product( $post_id );
		if ( ! $product ) {
			return;
		}

		// Save "Grants License"
		$grants_license = isset( $_POST['_uwlm_grants_license'] ) ? 'yes' : 'no';
		$product->update_meta_data( '_uwlm_grants_license', $grants_license );
		
		// Only save other data if it grants a license
		if ( 'yes' === $grants_license ) {
			$product->update_meta_data( '_uwlm_max_activations', isset( $_POST['_uwlm_max_activations'] ) ? absint( $_POST['_uwlm_max_activations'] ) : 1 );
			$product->update_meta_data( '_uwlm_validity_days', isset( $_POST['_uwlm_validity_days'] ) ? absint( $_POST['_uwlm_validity_days'] ) : 365 );
			$product->update_meta_data( '_uwlm_auto_generate', isset( $_POST['_uwlm_auto_generate'] ) ? 'yes' : 'no' );
			$product->update_meta_data( '_uwlm_current_version', isset( $_POST['_uwlm_current_version'] ) ? sanitize_text_field( $_POST['_uwlm_current_version'] ) : '' );
			$product->update_meta_data( '_uwlm_changelog', isset( $_POST['_uwlm_changelog'] ) ? wp_kses_post( $_POST['_uwlm_changelog'] ) : '' );
			
			// Handle file upload
			if ( ! empty( $_FILES['_uwlm_file_path']['name'] ) ) {
				$this->handle_file_upload( $product );
			}
		}
		
		$product->save();
	}

	/**
	 * Save the variation meta.
	 */
	public function save_variation_meta( $variation_id, $i ) {
		$variation = wc_get_product( $variation_id );
		if ( ! $variation ) {
			return;
		}

		// Save "Grants License"
		$grants_license = isset( $_POST['_uwlm_grants_license'][$i] ) ? 'yes' : 'no';
		$variation->update_meta_data( '_uwlm_grants_license', $grants_license );
		
		// Only save other data if it grants a license
		if ( 'yes' === $grants_license ) {
			$variation->update_meta_data( '_uwlm_max_activations', isset( $_POST['_uwlm_max_activations'][$i] ) ? absint( $_POST['_uwlm_max_activations'][$i] ) : 1 );
			$variation->update_meta_data( '_uwlm_validity_days', isset( $_POST['_uwlm_validity_days'][$i] ) ? absint( $_POST['_uwlm_validity_days'][$i] ) : 365 );
			$variation->update_meta_data( '_uwlm_auto_generate', isset( $_POST['_uwlm_auto_generate'][$i] ) ? 'yes' : 'no' );
		}
		
		$variation->save();
	}
	
	/**
	 * Handle secure file upload.
	 */
	private function handle_file_upload( $product ) {
		if ( ! function_exists( 'wp_handle_upload' ) ) {
			require_once( ABSPATH . 'wp-admin/includes/file.php' );
		}

		$file = $_FILES['_uwlm_file_path'];
		
		// Check file type
		$file_type = wp_check_filetype( $file['name'] );
		if( $file_type['ext'] !== 'zip' ) {
			// You could add an admin notice here
			return;
		}

		$upload_dir = wp_upload_dir();
		$uwlm_dir_path = $upload_dir['basedir'] . '/uwlm_files/' . $product->get_id();
		
		// Create secure directory
		if ( ! file_exists( $uwlm_dir_path ) ) {
			wp_mkdir_p( $uwlm_dir_path );
			// Add .htaccess and index.html for security
			file_put_contents( $uwlm_dir_path . '/.htaccess', 'deny from all' );
			file_put_contents( $uwlm_dir_path . '/index.html', '<!-- Silence is golden. -->' );
		}
		
		// Clean the version number to use as part of the filename
		$version = isset( $_POST['_uwlm_current_version'] ) ? sanitize_file_name( $_POST['_uwlm_current_version'] ) : '1.0.0';
		$filename = $product->get_id() . '-' . $version . '.zip';
		
		// Set destination
		$file_destination = $uwlm_dir_path . '/' . $filename;

		// Move the file
		if ( move_uploaded_file( $file['tmp_name'], $file_destination ) ) {
			// Save the file path to product meta
			$product->update_meta_data( '_uwlm_file_path', $file_destination );
		} else {
			// You could add an admin notice for upload failure
		}
	}

}