<?php
/**
 * Client-Side Plugin Updater Snippet.
 *
 * Instructions for your customers:
 * 1. Save this file as `plugin-updater.php` inside your plugin's main folder.
 * 2. In your main plugin file, add this line:
 * require_once( plugin_dir_path( __FILE__ ) . 'plugin-updater.php' );
 * 3. Initialize the class *after* your plugin is loaded:
 *
 * function my_plugin_init_updater() {
 * // Get the license key stored in settings
 * $license_key = get_option('my_plugin_license_key', '');
 *
 * new UWLM_Client_Updater(
 * 'https://YOUR-STORE-URL.com',  // <-- REPLACE with your store URL
 * __FILE__,                     // <-- Your main plugin file
 * array(
 * 'version'    => '1.0.0', // <-- The CURRENT version of the plugin
 * 'license'    => $license_key,
 * 'product_id' => '15627', // <-- The WooCommerce Product ID from your store
 * 'slug'       => 'my-plugin-slug' // <-- Your plugin's slug
 * )
 * );
 * }
 * add_action( 'plugins_loaded', 'my_plugin_init_updater' );
 *
 * 4. Make sure you have a settings page where the user can save their 'my_plugin_license_key'.
 */

if ( ! class_exists( 'UWLM_Client_Updater' ) ) :

	class UWLM_Client_Updater {

		private $api_url    = '';
		private $api_data   = array();
		private $plugin_file = '';
		private $slug        = '';

		public function __construct( $api_url, $plugin_file, $api_data = array() ) {
			$this->api_url     = trailingslashit( $api_url );
			$this->plugin_file = $plugin_file;
			$this->slug        = $api_data['slug'] ?? basename( $plugin_file, '.php' );
			$this->api_data    = wp_parse_args( $api_data, array(
				'version'    => '',
				'license'    => '',
				'product_id' => 0,
			) );

			// Hook into WordPress update checks
			add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'check_for_updates' ) );
			add_filter( 'plugins_api', array( $this, 'plugin_information_lightbox' ), 10, 3 );
		}

		/**
		 * Make the API request to check for a new version.
		 */
		private function api_request( $endpoint ) {
			$url = $this->api_url . 'wp-json/uwlm/v1/' . $endpoint;

			$params = array(
				'license_key' => $this->api_data['license'],
				'product_id'  => $this->api_data['product_id'],
			);
			
			// We use GET for version, which is public but keyed
			$options = get_option( 'uwlm_options', array() );
			$api_key    = $options['api_key'] ?? ''; // This is the API key from *your* store
			
			$response = wp_remote_get( $url, array(
				'timeout'   => 15,
				'headers'   => array(
					'X-Api-Key' => $api_key, // Use the public API key from your store
				),
				'body'      => $params
			) );

			if ( is_wp_error( $response ) ) {
				return false;
			}

			$body = wp_remote_retrieve_body( $response );
			$data = json_decode( $body );

			if ( $data && isset( $data->version ) ) {
				return $data;
			}
			
			return false;
		}

		/**
		 * Check for updates and insert into the update transient.
		 */
		public function check_for_updates( $transient ) {
			if ( empty( $transient->checked ) ) {
				return $transient;
			}

			// Don't check if license key is empty
			if ( empty( $this->api_data['license'] ) ) {
				return $transient;
			}
			
			$remote_version_info = $this->api_request( 'version' );
			
			if ( ! $remote_version_info ) {
				return $transient;
			}
			
			// If a new version is available, add it to the transient
			if ( version_compare( $remote_version_info->version, $this->api_data['version'], '>' ) ) {
				
				$plugin_data = new stdClass();
				$plugin_data->slug = $this->slug;
				$plugin_data->new_version = $remote_version_info->version;
				$plugin_data->package = $remote_version_info->package; // The secure download URL
				$plugin_data->url = $this->api_url; // Link to your store
				
				$transient->response[ $this->plugin_file ] = $plugin_data;
			}

			return $transient;
		}

		/**
		 * Handle the "View details" lightbox.
		 */
		public function plugin_information_lightbox( $res, $action, $args ) {
			// Check if it's our plugin
			if ( 'plugin_information' !== $action || $this->slug !== $args->slug ) {
				return $res;
			}
			
			$remote_info = $this->api_request( 'version' ); // Use 'version' as it has changelog
			
			if ( ! $remote_info ) {
				return $res;
			}

			$res = new stdClass();
			$res->name = $this->api_data['slug']; // You can add a 'name' field to api_data
			$res->slug = $this->slug;
			$res->version = $remote_info->version;
			$res->author = 'Your Plugin Author Name'; // Hardcode or add to api_data
			$res->homepage = $this->api_url;
			$res->download_link = $remote_info->package;
			$res->sections = array(
				'description' => 'Updates from ' . $this->api_url,
				'changelog'   => $remote_info->changelog,
			);

			return $res;
		}
	}

endif;