<?php
/**
 * API Class
 *
 * Registers and handles REST API endpoints.
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class UWLM_API {

	protected $namespace = 'uwlm/v1';

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Register the REST API routes.
	 */
	public function register_routes() {
		// POST /activate
		register_rest_route(
			$this->namespace,
			'/activate',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'handle_activate' ),
				'permission_callback' => array( $this, 'check_api_key' ),
				'args'                => array(
					'license_key'  => array( 'required' => true, 'type' => 'string' ),
					'product_id'   => array( 'required' => true, 'type' => 'string' ),
					'instance_url' => array( 'required' => true, 'type' => 'string' ),
				),
			)
		);

		// POST /deactivate
		register_rest_route(
			$this->namespace,
			'/deactivate',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'handle_deactivate' ),
				'permission_callback' => array( $this, 'check_api_key' ),
				'args'                => array(
					'license_key'  => array( 'required' => true, 'type' => 'string' ),
					'instance_url' => array( 'required' => true, 'type' => 'string' ),
				),
			)
		);

		// POST /check
		register_rest_route(
			$this->namespace,
			'/check',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'handle_check' ),
				'permission_callback' => array( $this, 'check_api_key' ),
				'args'                => array(
					'license_key' => array( 'required' => true, 'type' => 'string' ),
					'product_id'  => array( 'required' => true, 'type' => 'string' ),
				),
			)
		);
		
		// --- NEW: Download Endpoint ---
		register_rest_route(
			$this->namespace,
			'/download',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'handle_download' ),
				'permission_callback' => '__return_true', // Public, validation is in the callback
				'args'                => array(
					'key'        => array( 'required' => true, 'type' => 'string' ),
					'product_id' => array( 'required' => true, 'type' => 'string' ),
				),
			)
		);
	}

	/**
	 * Permission callback to check API keys.
	 *
	 * @param WP_REST_Request $request The request object.
	 * @return bool|WP_Error
	 */
	public function check_api_key( $request ) {
		$options    = get_option( 'uwlm_options', array() );
		$api_key    = $options['api_key'] ?? '';
		$api_secret = $options['api_secret'] ?? '';

		$req_key = $request->get_header( 'X-Api-Key' );
		
        // For activate/deactivate, require the secret
        $route = $request->get_route();
        if( strpos( $route, '/activate' ) !== false || strpos( $route, '/deactivate' ) !== false ) {
             $req_secret = $request->get_header( 'X-Api-Secret' );
             if ( empty( $api_secret ) || ! hash_equals( $api_secret, $req_secret ) ) {
                return new WP_Error( 'rest_forbidden', __( 'Invalid API Secret.', 'uwlm' ), array( 'status' => 401 ) );
             }
        }

        // For all routes, require the public key
		if ( empty( $api_key ) || ! hash_equals( $api_key, $req_key ) ) {
			return new WP_Error( 'rest_forbidden', __( 'Invalid API Key.', 'uwlm' ), array( 'status' => 401 ) );
		}

		return true;
	}

	/**
	 * Find and validate a license key.
	 *
	 * @param string $license_key Plaintext license key.
	 * @param string $product_id  Product ID.
	 * @return object|WP_Error The key object or WP_Error.
	 */
	private function validate_key( $license_key, $product_id ) {
		$hash = hash( 'sha256', $license_key );
		$key  = UWLM_DB::get_key_by_hash( $hash );

		if ( ! $key ) {
			return new WP_Error( 'invalid_key', __( 'License key does not exist.', 'uwlm' ), array( 'status' => 404 ) );
		}

		// Check product ID
        // Note: $product_id from API might be a string
		if ( $key->product_id != $product_id && $key->variation_id != $product_id ) {
			return new WP_Error( 'invalid_product', __( 'License key is not valid for this product.', 'uwlm' ), array( 'status' => 403 ) );
		}

		if ( 'active' !== $key->status ) {
			return new WP_Error( 'key_not_active', __( 'License key is not active.', 'uwlm' ), array( 'status' => 403 ) );
		}

		if ( $key->expires_at && strtotime( $key->expires_at ) < time() ) {
            // Set status to expired
            UWLM_DB::update_key_status( $key->id, 'expired' );
			return new WP_Error( 'key_expired', __( 'License key has expired.', 'uwlm' ), array( 'status' => 403 ) );
		}

		return $key;
	}

	/**
	 * Handle `POST /activate`
	 *
	 * @param WP_REST_Request $request
	 * @return WP_REST_Response|WP_Error
	 */
	public function handle_activate( $request ) {
		$license_key  = $request->get_param( 'license_key' );
		$product_id   = $request->get_param( 'product_id' );
		$instance_url = $request->get_param( 'instance_url' );

		$key = $this->validate_key( $license_key, $product_id );
		if ( is_wp_error( $key ) ) {
			return $key;
		}

		// Check if already activated
		$activations = UWLM_DB::get_activations( $key->id );
		foreach ( $activations as $activation ) {
			if ( $activation->instance_url === $instance_url ) {
				// Already active, just return success
				return new WP_REST_Response(
					array(
						'activated' => true,
						'expires'   => $key->expires_at,
						'message'   => __( 'License already active for this instance.', 'uwlm' ),
					),
					200
				);
			}
		}

		// Check activation limit
		if ( $key->max_activations > 0 && $key->activations >= $key->max_activations ) {
			return new WP_Error( 'max_activations', __( 'Maximum activations reached.', 'uwlm' ), array( 'status' => 403 ) );
		}

		// Add activation
		$added = UWLM_DB::add_activation( $key->id, $instance_url );

		if ( ! $added ) {
			return new WP_Error( 'activation_failed', __( 'Could not save activation.', 'uwlm' ), array( 'status' => 500 ) );
		}

		return new WP_REST_Response(
			array(
				'activated' => true,
				'expires'   => $key->expires_at,
				'message'   => __( 'License activated.', 'uwlm' ),
			),
			200
		);
	}

	/**
	 * Handle `POST /deactivate`
	 *
	 * @param WP_REST_Request $request
	 * @return WP_REST_Response|WP_Error
	 */
	public function handle_deactivate( $request ) {
		$license_key  = $request->get_param( 'license_key' );
		$instance_url = $request->get_param( 'instance_url' );

		$hash = hash( 'sha256', $license_key );
		$key  = UWLM_DB::get_key_by_hash( $hash );

		if ( ! $key ) {
			return new WP_Error( 'invalid_key', __( 'License key does not exist.', 'uwlm' ), array( 'status' => 404 ) );
		}

		$removed = UWLM_DB::remove_activation( $key->id, $instance_url );

		if ( ! $removed ) {
			return new WP_Error( 'deactivation_failed', __( 'Instance not found or could not be removed.', 'uwlm' ), array( 'status' => 404 ) );
		}

		return new WP_REST_Response(
			array(
				'deactivated' => true,
				'message'     => __( 'License deactivated.', 'uwlm' ),
			),
			200
		);
	}

	/**
	 * Handle `POST /check`
	 *
	 * @param WP_REST_Request $request
	 * @return WP_REST_Response|WP_Error
	 */
	public function handle_check( $request ) {
		$license_key = $request->get_param( 'license_key' );
		$product_id  = $request->get_param( 'product_id' );

		$key = $this->validate_key( $license_key, $product_id );
		if ( is_wp_error( $key ) ) {
			return $key;
		}

		// Key is valid
		$activations_left = 'unlimited';
		if ( $key->max_activations > 0 ) {
			$activations_left = $key->max_activations - $key->activations;
		}

		return new WP_REST_Response(
			array(
				'valid'              => true,
				'expires'            => $key->expires_at,
				'activations_left'   => $activations_left,
                'activations_max'    => $key->max_activations > 0 ? $key->max_activations : 'unlimited',
				'activations_current' => (int) $key->activations,
			),
			200
		);
	}

	/**
	 * Handle `GET /version`
	 *
	 * @param WP_REST_Request $request
	 * @return WP_REST_Response|WP_Error
	 */
	public function handle_version( $request ) {
		$license_key = $request->get_param( 'license_key' );
		$product_id  = $request->get_param( 'product_id' ); // This is the WC Product ID

		$key = $this->validate_key( $license_key, $product_id );
		if ( is_wp_error( $key ) ) {
			return $key;
		}
        
        // Get the parent product ID if it's a variation
        $parent_product_id = $key->variation_id ? $key->product_id : $key->product_id;
        $product = wc_get_product($parent_product_id);
        
        if(!$product) {
            return new WP_Error( 'product_not_found', __( 'Associated product not found.', 'uwlm' ), array( 'status' => 404 ) );
        }

		// Key is valid, get version info from the *main product*
		$version   = $product->get_meta( '_uwlm_current_version' );
		$changelog = $product->get_meta( '_uwlm_changelog' );
		$file_hash = $product->get_meta( '_uwlm_file_hash' ); // --- NEW ---
        
        // --- UPDATED ---
        // Get download URL if the product is downloadable
        // Generate a secure, license-key-based download URL
		$download_url = home_url( sprintf(
			'/wp-json/%s/download?key=%s&product_id=%s',
			$this->namespace,
			urlencode( $license_key ),
			urlencode( $product_id ) // Use the ID (simple or var) that was requested
		) );

		return new WP_REST_Response(
			array(
				'version'      => $version,
				'changelog'    => $changelog,
				'download_url' => $download_url,
				'package'      => $download_url, // 'package' is what the WP updater expects
				'file_hash'    => $file_hash,
			),
			200
		);
	}
	
	/**
	 * --- NEW ---
	 * Handle `GET /download`
	 *
	 * @param WP_REST_Request $request
	 * @return void|WP_Error
	 */
	public function handle_download( $request ) {
		$license_key = $request->get_param( 'key' );
		$product_id  = $request->get_param( 'product_id' );

		// 1. Validate the license key
		$key = $this->validate_key( $license_key, $product_id );
		if ( is_wp_error( $key ) ) {
			$error_code = $key->get_error_code();
			$error_message = $key->get_error_message();
			wp_die( '<div>' . esc_html( $error_message ) . '</div>', esc_html( $error_code ), array( 'status' => 403 ) );
		}

		// 2. Key is valid. Get the file path from the parent product.
		$parent_product_id = $key->variation_id ? $key->product_id : $key->product_id;
		$file_path = get_post_meta( $parent_product_id, '_uwlm_file_path', true );
		$version = get_post_meta( $parent_product_id, '_uwlm_current_version', true );
		$product = wc_get_product( $parent_product_id );

		if ( empty( $file_path ) || ! file_exists( $file_path ) || ! $product ) {
			wp_die( 'File not found for this product. Please contact support.', 'File Error', array( 'status' => 404 ) );
		}
		
		// 3. Get the product slug for a better filename
		$file_name = $product->get_slug() . '-' . $version . '.zip';

		// 4. Stream the file securely
		header( 'Content-Description: File Transfer' );
		header( 'Content-Type: application/zip' );
		header( 'Content-Disposition: attachment; filename="' . sanitize_file_name( $file_name ) . '"' );
		header( 'Content-Transfer-Encoding: binary' );
		header( 'Expires: 0' );
		header( 'Cache-Control: must-revalidate' );
		header( 'Pragma: public' );
		header( 'Content-Length: ' . filesize( $file_path ) );
		
		// Clear output buffer
		if ( ob_get_level() ) {
			ob_end_clean();
		}
		
		readfile( $file_path );
		exit;
	}
}