<?php
/**
 * DB Class
 *
 * Handles activation and database table creation.
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class UWLM_DB {

	/**
	 * Plugin activation hook.
	 * Creates custom database tables.
	 */
	public static function activate() {
		global $wpdb;
		$wpdb->hide_errors();

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$collate = $wpdb->get_charset_collate();

		// Table for License Keys
		$table_name_keys = $wpdb->prefix . UWLM_LICENSE_KEYS_TABLE;
		$sql_keys_table = "CREATE TABLE {$table_name_keys} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			license_key longtext NOT NULL,
			iv varchar(255) NOT NULL,
			key_hash varchar(64) NOT NULL,
			order_id bigint(20) unsigned NOT NULL,
			product_id bigint(20) unsigned NOT NULL,
			variation_id bigint(20) unsigned DEFAULT 0 NOT NULL,
			customer_id bigint(20) unsigned NOT NULL,
			activations int(11) DEFAULT 0 NOT NULL,
			max_activations int(11) DEFAULT 1 NOT NULL,
			valid_for int(11) DEFAULT 0 NOT NULL,
			created_at datetime NOT NULL,
			expires_at datetime DEFAULT NULL,
			status varchar(20) DEFAULT 'inactive' NOT NULL,
			PRIMARY KEY  (id),
			KEY order_id (order_id),
			KEY product_id (product_id),
			KEY customer_id (customer_id),
			KEY status (status),
			KEY key_hash (key_hash)
		) $collate;";

		dbDelta( $sql_keys_table );

		// Table for Activations
		$table_name_activations = $wpdb->prefix . UWLM_ACTIVATIONS_TABLE;
		$sql_activations_table = "CREATE TABLE {$table_name_activations} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			license_id bigint(20) unsigned NOT NULL,
			instance_url varchar(255) NOT NULL,
			activated_at datetime NOT NULL,
			PRIMARY KEY  (id),
			KEY license_id (license_id),
			KEY instance_url (instance_url)
		) $collate;";

		dbDelta( $sql_activations_table );
	}

	// --- Helper functions to interact with the DB ---

	/**
	 * Get a license key by its hash.
	 *
	 * @param string $hash The sha256 hash of the plaintext license key.
	 * @return object|null
	 */
	public static function get_key_by_hash( $hash ) {
		global $wpdb;
		$table = $wpdb->prefix . UWLM_LICENSE_KEYS_TABLE;
		return $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table} WHERE key_hash = %s", $hash ) );
	}

	/**
	 * Get a license key by its ID.
	 *
	 * @param int $key_id The ID of the license key.
	 * @return object|null
	 */
	public static function get_key_by_id( $key_id ) {
		global $wpdb;
		$table = $wpdb->prefix . UWLM_LICENSE_KEYS_TABLE;
		return $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table} WHERE id = %d", $key_id ) );
	}

	/**
	 * Add an activation instance.
	 *
	 * @param int    $license_id   The ID of the license key.
	 * @param string $instance_url The URL of the activated instance.
	 * @return bool
	 */
	public static function add_activation( $license_id, $instance_url ) {
		global $wpdb;
		$table = $wpdb->prefix . UWLM_ACTIVATIONS_TABLE;
		
		$result = $wpdb->insert(
			$table,
			array(
				'license_id'   => $license_id,
				'instance_url' => esc_url_raw( $instance_url ),
				'activated_at' => current_time( 'mysql' ),
			),
			array( '%d', '%s', '%s' )
		);
		
		if($result) {
			self::update_activation_count($license_id);
			return true;
		}
		return false;
	}

	/**
	 * Remove an activation instance.
	 *
	 * @param int    $license_id   The ID of the license key.
	 * @param string $instance_url The URL of the activated instance.
	 * @return bool
	 */
	public static function remove_activation( $license_id, $instance_url ) {
		global $wpdb;
		$table = $wpdb->prefix . UWLM_ACTIVATIONS_TABLE;

		$result = $wpdb->delete(
			$table,
			array(
				'license_id'   => $license_id,
				'instance_url' => esc_url_raw( $instance_url ),
			),
			array( '%d', '%s' )
		);

		if($result) {
			self::update_activation_count($license_id);
			return true;
		}
		return false;
	}

	/**
	 * Get all activations for a license.
	 *
	 * @param int $license_id The ID of the license key.
	 * @return array
	 */
	public static function get_activations( $license_id ) {
		global $wpdb;
		$table = $wpdb->prefix . UWLM_ACTIVATIONS_TABLE;
		return $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} WHERE license_id = %d", $license_id ) );
	}

	/**
	 * Update the 'activations' count in the main keys table.
	 *
	 * @param int $license_id The ID of the license key.
	 */
	public static function update_activation_count( $license_id ) {
		global $wpdb;
		$keys_table = $wpdb->prefix . UWLM_LICENSE_KEYS_TABLE;
		$activations_table = $wpdb->prefix . UWLM_ACTIVATIONS_TABLE;

		// Count current activations
		$count = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(id) FROM {$activations_table} WHERE license_id = %d", $license_id ) );

		// Update keys table
		$wpdb->update(
			$keys_table,
			array( 'activations' => $count ),
			array( 'id' => $license_id ),
			array( '%d' ),
			array( '%d' )
		);
	}
    
    /**
	 * Update a key's status.
	 *
	 * @param int    $license_id The ID of the license key.
	 * @param string $status     The new status.
	 */
    public static function update_key_status( $license_id, $status ) {
        global $wpdb;
		$keys_table = $wpdb->prefix . UWLM_LICENSE_KEYS_TABLE;

        $wpdb->update(
			$keys_table,
			array( 'status' => $status ),
			array( 'id' => $license_id ),
			array( '%s' ),
			array( '%d' )
		);
    }
}