<?php
/**
 * Email Class
 *
 * Handles sending emails with license keys.
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class UWLM_Email {

	/**
	 * Send the license key email to the customer.
	 *
	 * @param WC_Order $order          The order object.
	 * @param array    $generated_keys Array of keys that were generated.
	 */
	public static function send_license_key_email( $order, $generated_keys ) {
		$mailer    = WC()->mailer();
		$recipient = $order->get_billing_email();
		$subject   = apply_filters( 'uwlm_email_subject', __( 'Your License Key(s)', 'uwlm' ), $order );
		$heading   = apply_filters( 'uwlm_email_heading', __( 'Your License Key(s)', 'uwlm' ), $order );
		
		ob_start();
		self::email_template( $order, $generated_keys, $heading );
		$message = ob_get_clean();

		$headers = "Content-Type: text/html\r\n";

		$mailer->send( $recipient, $subject, $message, $headers, array() );
	}

	/**
	 * Simple email template.
	 *
	 * @param WC_Order $order
	 * @param array    $generated_keys
	 * @param string   $heading
	 */
	private static function email_template( $order, $generated_keys, $heading ) {
		// This can be replaced with a full WC Email Template
		?>
		<!DOCTYPE html>
		<html>
		<head>
			<meta http-equiv="Content-Type" content="text/html; charset=utf-8" />
			<title><?php echo esc_html( $heading ); ?></title>
		</head>
		<body style="font-family: Arial, sans-serif; margin: 0; padding: 20px; background-color: #f4f4f4;">
			<div style="max-width: 600px; margin: 0 auto; background-color: #ffffff; border: 1px solid #e0e0e0; border-radius: 5px; padding: 30px;">
				<h1 style="font-size: 24px; color: #333; margin-top: 0;"><?php echo esc_html( $heading ); ?></h1>
				<p style="font-size: 16px; color: #555; line-height: 1.6;">
					<?php
					echo sprintf(
						esc_html__( 'Hi %s,', 'uwlm' ),
						esc_html( $order->get_billing_first_name() )
					);
					?>
				</p>
				<p style="font-size: 16px; color: #555; line-height: 1.6;">
					<?php esc_html_e( 'Thank you for your purchase. Here are your license key(s):', 'uwlm' ); ?>
				</p>

				<table style="width: 100%; border-collapse: collapse; margin-top: 20px; margin-bottom: 20px;">
					<thead>
						<tr>
							<th style="border: 1px solid #ddd; padding: 10px; text-align: left; background-color: #f9f9f9;"><?php esc_html_e( 'Product', 'uwlm' ); ?></th>
							<th style="border: 1px solid #ddd; padding: 10px; text-align: left; background-color: #f9f9f9;"><?php esc_html_e( 'License Key', 'uwlm' ); ?></th>
							<th style="border: 1px solid #ddd; padding: 10px; text-align: left; background-color: #f9f9f9;"><?php esc_html_e( 'Expires', 'uwlm' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php foreach ( $generated_keys as $key_data ) : ?>
							<tr>
								<td style="border: 1px solid #ddd; padding: 10px;"><?php echo esc_html( $key_data['product_name'] ); ?></td>
								<td style="border: 1px solid #ddd; padding: 10px;">
									<code style="font-family: monospace, courier; background-color: #eee; padding: 3px 5px; border-radius: 3px;"><?php echo esc_html( $key_data['plaintext_key'] ); ?></code>
								</td>
								<td style="border: 1px solid #ddd; padding: 10px;">
									<?php echo $key_data['expires_at'] ? esc_html( date_i18n( get_option( 'date_format' ), strtotime( $key_data['expires_at'] ) ) ) : esc_html__( 'Lifetime', 'uwlm' ); ?>
								</td>
							</tr>
						<?php endforeach; ?>
					</tbody>
				</table>

				<p style="font-size: 16px; color: #555; line-height: 1.6;">
					<?php esc_html_e( 'You can also find your keys in your "My Account" page.', 'uwlm' ); ?>
				</p>
				<p style="font-size: 16px; color: #555; line-height: 1.6;">
					<?php esc_html_e( 'Thanks,', 'uwlm' ); ?><br>
					<?php echo esc_html( get_bloginfo( 'name' ) ); ?>
				</p>
			</div>
		</body>
		</html>
		<?php
	}
}