<?php
/**
 * Helpers Class
 *
 * Provides static helper methods for encryption and key generation.
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class UWLM_Helpers {

	/**
	 * Get the encryption key.
	 *
	 * Tries to use WP_SALTS, falls back to a stored option.
	 *
	 * @return string The encryption key.
	 */
	private static function get_encryption_key() {
		if ( defined( 'AUTH_KEY' ) && '' !== AUTH_KEY ) {
			return hash( 'sha256', AUTH_KEY . SECURE_AUTH_KEY );
		}

		// Fallback to a key stored in options
		$key = get_option( 'uwlm_encryption_key' );
		if ( ! $key ) {
			$key = bin2hex( random_bytes( 32 ) );
			update_option( 'uwlm_encryption_key', $key );
		}
		return $key;
	}

	/**
	 * Encrypt a license key.
	 *
	 * @param string $key_string The plaintext license key.
	 * @return array|false An array containing 'key' and 'iv', or false on failure.
	 */
	public static function encrypt_key( $key_string ) {
		if ( ! function_exists( 'openssl_encrypt' ) ) {
			error_log( 'UWLM Error: openssl_encrypt function does not exist. Cannot encrypt keys.' );
			return false;
		}

		$key = self::get_encryption_key();
		$ivlen = openssl_cipher_iv_length( 'aes-256-cbc' );
		$iv = openssl_random_pseudo_bytes( $ivlen );
		$encrypted_key = openssl_encrypt( $key_string, 'aes-256-cbc', $key, 0, $iv );

		if ( false === $encrypted_key ) {
			return false;
		}

		return array(
			'key' => base64_encode( $encrypted_key ),
			'iv'  => base64_encode( $iv ),
		);
	}

	/**
	 * Decrypt a license key.
	 *
	 * @param string $encrypted_key The base64 encoded encrypted key.
	 * @param string $iv            The base64 encoded initialization vector.
	 * @return string|false The plaintext key or false on failure.
	 */
	public static function decrypt_key( $encrypted_key, $iv ) {
		if ( ! function_exists( 'openssl_decrypt' ) ) {
			error_log( 'UWLM Error: openssl_decrypt function does not exist. Cannot decrypt keys.' );
			return false;
		}
		
		$key = self::get_encryption_key();
		$encrypted_key = base64_decode( $encrypted_key );
		$iv = base64_decode( $iv );

		if ( false === $encrypted_key || false === $iv ) {
			return false;
		}

		return openssl_decrypt( $encrypted_key, 'aes-256-cbc', $key, 0, $iv );
	}

	/**
	 * Generate a cryptographically strong, unique license key string.
	 *
	 * @return string The formatted license key.
	 */
	public static function generate_key_string() {
		$random_string = bin2hex( random_bytes( 10 ) ); // 20 chars
		$key           = strtoupper( implode( '-', str_split( $random_string, 5 ) ) );
		return $key;
	}

}