<?php
/**
 * Handles automatic license key generation upon WooCommerce order completion.
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class UWLM_Order_Automation {

	private $generated_keys_for_email = array();

	/**
	 * Constructor.
	 */
	public function __construct() {
		// Get the option from settings
		$options      = get_option( 'uwlm_options', array() );
		$order_status = $options['order_status'] ?? 'wc-completed';
		$hook_name    = "woocommerce_order_status_{$order_status}";

		// Use the selected order status hook
		add_action( $hook_name, array( $this, 'generate_license_keys_for_order' ), 10, 1 );

		// Hook to send email *after* all keys are generated
		add_action( 'woocommerce_order_status_changed', array( $this, 'send_license_key_email_on_status_change' ), 99, 4 );
	}

	/**
	 * Generates license keys for all products in an order.
	 *
	 * @param int $order_id The ID of the completed order.
	 */
	public function generate_license_keys_for_order( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			return;
		}

		// Prevent duplicate generation
		if ( $order->get_meta( '_uwlm_keys_generated' ) === 'yes' ) {
			return;
		}

		$customer_id            = $order->get_customer_id();
		$this->generated_keys_for_email = array(); // Clear for this order

		foreach ( $order->get_items() as $item_id => $item ) {
			$product_id   = $item->get_product_id();
			$variation_id = $item->get_variation_id();
			$quantity     = $item->get_quantity();
			
			// Use variation ID if it exists, otherwise product ID
			$license_product_id = $variation_id ? $variation_id : $product_id;
			$product = wc_get_product( $license_product_id );

			if ( ! $product ) {
				continue;
			}
			
			// Check if this product/variation grants a license
			$grants_license = $product->get_meta( '_uwlm_grants_license' );
			if ( 'yes' !== $grants_license ) {
				continue;
			}

			// --- Apply Free/Premium Restrictions ---
			// ** THIS IS THE FIX **
			// Swapped non-existent class `UWLM_Licensing_Manager` with global function `uwlm_is_premium()`
			if ( ! uwlm_is_premium() ) {
				// Check if we can generate this license (is it the first one?)
				global $wpdb;
				$count = $wpdb->get_var(
					"SELECT COUNT(DISTINCT post_id)
					FROM $wpdb->postmeta
					WHERE meta_key = '_uwlm_grants_license' AND meta_value = 'yes'"
				);
				
				$parent_product_id = $product->is_type('variation') ? $product->get_parent_id() : $product->get_id();
				$was_already_licensed = get_post_meta($parent_product_id, '_uwlm_grants_license', true) === 'yes';

				if ( (int) $count > 1 && ! $was_already_licensed ) {
					// Log that license gen was skipped
					$order->add_order_note( sprintf(
						__( 'License key generation for "%s" skipped: Free version limit reached.', 'uwlm' ),
						$product->get_name()
					) );
					continue; // Skip generation
				}
			}
			
			// Get license settings
			$max_activations = $product->get_meta( '_uwlm_max_activations' ) ?: 1;
			$validity_days   = $product->get_meta( '_uwlm_validity_days' ) ?: 365;
			$auto_generate   = $product->get_meta( '_uwlm_auto_generate' ) ?: 'yes';

			if ( 'yes' !== $auto_generate ) {
				// TODO: Implement logic to pull from a pre-made list
				continue;
			}

			// Generate a key for each quantity
			for ( $i = 0; $i < $quantity; $i++ ) {
				$license_key = UWLM_Helpers::generate_key_string();
				$encryption  = UWLM_Helpers::encrypt_key( $license_key );

				if ( ! $encryption ) {
					error_log( 'UWLM: Failed to encrypt license key for product ' . $product_id . ' in order ' . $order_id );
					continue;
				}

				$expires_at = null;
				if ( $validity_days > 0 ) {
					$expires_at = date( 'Y-m-d H:i:s', strtotime( '+' . $validity_days . ' days' ) );
				}
				
				$parent_product_id = $product->is_type('variation') ? $product->get_parent_id() : $product_id;

				// Use our helper to insert the key
				$inserted_id = UWLM_Admin_Actions::insert_manual_key(
					$license_key,
					$parent_product_id,
					$variation_id,
					$max_activations,
					$validity_days,
					$expires_at,
					'active',
					$customer_id,
					$order_id
				);
				
				if ( $inserted_id ) {
					// Store for email
					$this->generated_keys_for_email[] = array(
						'product_name'  => $product->get_name(),
						'plaintext_key' => $license_key,
						'expires_at'    => $expires_at,
					);
				} else {
					error_log( 'UWLM: Failed to insert license key into DB for product ' . $product_id . ' in order ' . $order_id . '. Error: ' . $wpdb->last_error );
				}
			}
		}
		
		// Mark order as processed
		$order->update_meta_data( '_uwlm_keys_generated', 'yes' );
		$order->save();
	}

	/**
	 * Send the consolidated email after the status has fully changed.
	 */
	public function send_license_key_email_on_status_change( $order_id, $status_from, $status_to, $order ) {
		// Get the trigger status from settings
		$options      = get_option( 'uwlm_options', array() );
		$trigger_status = $options['order_status'] ?? 'wc-completed';
		
		// Check if the new status matches our trigger status
		if ( 'wc-' . $status_to !== $trigger_status && $status_to !== $trigger_status ) {
			return;
		}

		// Check if we have keys to send
		if ( empty( $this->generated_keys_for_email ) ) {
			// This can happen if the hook ran but no keys were generated (e.g., non-licensed products)
			// Let's re-check the order items just in case
			if ( $order->get_meta( '_uwlm_keys_generated' ) === 'yes' ) {
				// Keys were generated, but maybe on a previous run. Let's not re-send.
				return;
			}
			
			// Manually run the generation logic again if it hasn't run
			$this->generate_license_keys_for_order( $order_id );
			
			// If still no keys, abort.
			if( empty( $this->generated_keys_for_email ) ) {
				return;
			}
		}

		// Send email
		UWLM_Email::send_license_key_email( $order, $this->generated_keys_for_email );
		
		// Clear the array
		$this->generated_keys_for_email = array();
	}
}