<?php
/**
 * Public Class
 *
 * Handles "My Account" page functionality.
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class UWLM_Public {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'add_my_account_endpoint' ) );
		add_filter( 'woocommerce_account_menu_items', array( $this, 'add_my_account_menu_item' ) );
		add_action( 'woocommerce_account_my-licenses_endpoint', array( $this, 'render_my_licenses_content' ) );

		// AJAX for deactivation
		add_action( 'wp_ajax_uwlm_deactivate_instance', array( $this, 'ajax_deactivate_instance' ) );
        
        // Enqueue scripts
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
	}

	/**
	 * Add "My Licenses" endpoint.
	 */
	public function add_my_account_endpoint() {
		add_rewrite_endpoint( 'my-licenses', EP_ROOT | EP_PAGES );
	}

	/**
	 * Add "My Licenses" to the "My Account" menu.
	 *
	 * @param array $items The existing menu items.
	 * @return array The modified menu items.
	 */
	public function add_my_account_menu_item( $items ) {
		$logout = $items['customer-logout'];
		unset( $items['customer-logout'] );

		$items['my-licenses'] = __( 'My Licenses', 'uwlm' );

		$items['customer-logout'] = $logout;
		return $items;
	}
    
    /**
     * Enqueue scripts for "My Account" page.
     */
    public function enqueue_scripts() {
        if ( is_account_page() && is_wc_endpoint_url( 'my-licenses' ) ) {
            wp_enqueue_style( 'uwlm-public-css', UWLM_PLUGIN_URL . 'public/css/uwlm-public.css', array(), UWLM_VERSION );
            wp_enqueue_script( 'uwlm-public-js', UWLM_PLUGIN_URL . 'public/js/uwlm-public.js', array( 'jquery' ), UWLM_VERSION, true );
            wp_localize_script( 'uwlm-public-js', 'uwlm_public', array(
                'ajax_url' => admin_url( 'admin-ajax.php' ),
                'nonce'    => wp_create_nonce( 'uwlm_deactivate_nonce' ),
                'confirm_deactivation' => __( 'Are you sure you want to deactivate this instance?', 'uwlm' ),
                'error_message' => __( 'An error occurred. Please try again.', 'uwlm' ),
            ) );
        }
    }

	/**
	 * Render the content for the "My Licenses" page.
	 */
	public function render_my_licenses_content() {
		global $wpdb;
		$table     = $wpdb->prefix . UWLM_LICENSE_KEYS_TABLE;
		$customer_id = get_current_user_id();

		$keys = $wpdb->get_results( $wpdb->prepare(
			"SELECT * FROM {$table} WHERE customer_id = %d ORDER BY created_at DESC",
			$customer_id
		) );

		if ( empty( $keys ) ) {
			echo '<p>' . esc_html__( 'You do not have any license keys yet.', 'uwlm' ) . '</p>';
			return;
		}
		
		wc_get_template( 'my-licenses-table.php', array( 'keys' => $keys ), '', UWLM_PLUGIN_DIR . 'public/templates/' );
	}

	/**
	 * Handle AJAX request to deactivate an instance.
	 */
	public function ajax_deactivate_instance() {
		check_ajax_referer( 'uwlm_deactivate_nonce', 'nonce' );

		$license_id   = isset( $_POST['license_id'] ) ? absint( $_POST['license_id'] ) : 0;
		$instance_url = isset( $_POST['instance_url'] ) ? esc_url_raw( $_POST['instance_url'] ) : '';

		if ( ! $license_id || ! $instance_url ) {
			wp_send_json_error( array( 'message' => __( 'Invalid request.', 'uwlm' ) ) );
		}

		// Verify ownership
		$key = UWLM_DB::get_key_by_id( $license_id );
		if ( ! $key || $key->customer_id != get_current_user_id() ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'uwlm' ) ) );
		}

		// Remove the activation
		$removed = UWLM_DB::remove_activation( $license_id, $instance_url );

		if ( $removed ) {
            // Get updated key data
            $updated_key = UWLM_DB::get_key_by_id( $license_id );
			wp_send_json_success( array( 
                'message' => __( 'Instance deactivated.', 'uwlm' ),
                'activations_html' => sprintf( '%d / %s', $updated_key->activations, $updated_key->max_activations > 0 ? $updated_key->max_activations : '∞' )
            ) );
		} else {
			wp_send_json_error( array( 'message' => __( 'Could not deactivate instance.', 'uwlm' ) ) );
		}
	}
}