<?php
/**
 * Plugin Name:         Ultimate WooCommerce Licenses Manager
 * Plugin URI:          https://ultimatemultimediaconsult.com/tools/
 * Description:         Sell and manage software licenses, versions, and activations through WordPress & WooCommerce.
 * Version:             1.1.3
 * Author:              Gerald Businge
 * Author URI:          https://geraldbusinge.com/tools
 * License:             GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       uwlm
 * Domain Path:       /languages
 * WC requires at least: 6.0
 * WC tested up to: 8.8
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Define Constants
define( 'UWLM_VERSION', '1.1.2' );
define( 'UWLM_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'UWLM_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'UWLM_LICENSE_KEYS_TABLE', 'uwlm_license_keys' );
define( 'UWLM_ACTIVATIONS_TABLE', 'uwlm_license_activations' );

// --- NEW ---
// These constants are for your *own* store where you sell this plugin
define( 'UWLM_PRO_STORE_URL', 'https://www.ultimatemultimediaconsult.com' );
define( 'UWLM_PRO_PRODUCT_ID', 15627 ); // The product ID of this plugin on your store

/**
 * --- NEW / MOVED ---
 * Helper function to check if the Pro version is active.
 * This is placed in the main file to be globally available.
 *
 * @return bool
 */
function uwlm_is_premium() {
	$options = get_option( 'uwlm_options', array() );
	$status  = $options['premium_status'] ?? 'inactive';
	return $status === 'active';
}

/**
 * Main Plugin Class
 */
final class Ultimate_WooCommerce_Licenses_Manager {

	/**
	 * The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Main Instance.
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->includes();
		$this->init_hooks();
	}

	/**
	 * Include required files.
	 */
	public function includes() {
		// Core
		require_once UWLM_PLUGIN_DIR . 'includes/class-uwlm-db.php';
		require_once UWLM_PLUGIN_DIR . 'includes/class-uwlm-helpers.php';
		require_once UWLM_PLUGIN_DIR . 'includes/class-uwlm-order-automation.php';
		require_once UWLM_PLUGIN_DIR . 'includes/class-uwlm-email.php';
		
		// API
		require_once UWLM_PLUGIN_DIR . 'includes/api/class-uwlm-api.php';
		
		if ( is_admin() ) {
			// Admin
			require_once UWLM_PLUGIN_DIR . 'admin/class-uwlm-admin.php';
			require_once UWLM_PLUGIN_DIR . 'admin/class-uwlm-product-settings.php';
			require_once UWLM_PLUGIN_DIR . 'admin/class-uwlm-license-keys-list-table.php';
			// --- NEW ---
			// Load Admin Actions earlier to make helper functions available
			require_once UWLM_PLUGIN_DIR . 'admin/class-uwlm-admin-actions.php';
		}
		
		if ( ! is_admin() || ( defined( 'DOING_AJAX' ) && DOING_AJAX ) ) {
			// Public
			require_once UWLM_PLUGIN_DIR . 'public/class-uwlm-public.php';
		}
	}

	/**
	 * Hook into actions and filters.
	 */
	private function init_hooks() {
		// Activation hook to create DB tables
		register_activation_hook( __FILE__, array( 'UWLM_DB', 'create_tables' ) );
		
		// HPOS Compatibility
		add_action( 'before_woocommerce_init', array( $this, 'declare_hpos_compatibility' ) );

		// Init classes
		if ( is_admin() ) {
			new UWLM_Admin();
			new UWLM_Product_Settings();
			UWLM_Admin_Actions::init(); // Init static class
		}
		
		if ( ! is_admin() || ( defined( 'DOING_AJAX' ) && DOING_AJAX ) ) {
			new UWLM_Public();
		}

		new UWLM_Order_Automation();
		new UWLM_Email();
		new UWLM_API();
	}

	/**
	 * Declare HPOS compatibility.
	 */
	public function declare_hpos_compatibility() {
		if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
		}
	}
}

/**
 * Begins execution of the plugin.
 */
function UWLM_run() {
	return Ultimate_WooCommerce_Licenses_Manager::instance();
}
UWLM_run();