<?php
/**
 * Admin Dashboard Manager
 *
 * Handles the backend UI, settings pages, and module toggles.
 *
 * @package    Ultimate_WP_SCT
 * @subpackage Ultimate_WP_SCT/inc/admin
 */

namespace UWSCT\Inc\Admin;

class Admin_Manager {

	/**
	 * Licensing Manager Instance.
	 *
	 * @var \UWSCT\Inc\Common\Licensing
	 */
	private $licensing;

	/**
	 * Current active tab.
	 *
	 * @var string
	 */
	private $active_tab;

	/**
	 * Initialize the class.
	 *
	 * @param \UWSCT\Inc\Common\Licensing $licensing Dependency injection.
	 */
	public function __construct( $licensing ) {
		$this->licensing = $licensing;
		$this->active_tab = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'dashboard';
	}

	/**
	 * Register the administration menu for this plugin.
	 */
	public function add_plugin_admin_menu() {
		add_menu_page(
			__( 'Ultimate WP SCT', 'ultimate-wp-sct' ),
			__( 'Ultimate WP SCT', 'ultimate-wp-sct' ),
			'manage_options',
			'uwsct_dashboard',
			[ $this, 'display_plugin_dashboard' ],
			'dashicons-shield-alt', // Main Icon
			2
		);
	}

	/**
	 * Register and enqueue admin-specific styles and scripts.
	 */
	public function enqueue_styles() {
		if ( isset( $_GET['page'] ) && 'uwsct_dashboard' === $_GET['page'] ) {
			wp_enqueue_style( 'uwsct-admin-css', UWSCT_PLUGIN_URL . 'assets/css/admin.css', [], UWSCT_VERSION, 'all' );
		}
	}

	/**
	 * Register and enqueue admin-specific JavaScript.
	 */
	public function enqueue_scripts() {
		if ( isset( $_GET['page'] ) && 'uwsct_dashboard' === $_GET['page'] ) {
			wp_enqueue_script( 'uwsct-admin-js', UWSCT_PLUGIN_URL . 'assets/js/admin.js', [ 'jquery' ], UWSCT_VERSION, true );
			// Enqueue AI Logic
			wp_enqueue_script( 'uwsct-admin-ai-js', UWSCT_PLUGIN_URL . 'assets/js/admin-ai.js', [ 'jquery', 'uwsct-admin-js' ], UWSCT_VERSION, true );
			
			// Localize script for AJAX
			wp_localize_script( 'uwsct-admin-js', 'uwsct_vars', [
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'nonce'    => wp_create_nonce( 'uwsct_admin_nonce' ),
			]);
		}
	}

	/**
	 * Render the dashboard view.
	 */
	public function display_plugin_dashboard() {
		// Check License/Trial Status first
		$is_active = $this->licensing->is_license_active_or_trial();
		$days_left = $this->licensing->get_trial_days_remaining();

		// Load Header
		$this->safe_include( 'inc/admin/views/header.php', [ 'is_active' => $is_active, 'days_left' => $days_left ] );

		// Load Tab Content
		echo '<div class="uwsct-container">';
		
		// Sidebar Navigation
		$this->safe_include( 'inc/admin/views/sidebar-nav.php' );

		// Main Content Area
		echo '<div class="uwsct-content">';
		
		if ( ! $is_active && 'license' !== $this->active_tab ) {
			// Block access if trial expired (except for license tab)
			$this->safe_include( 'inc/admin/views/lockout-screen.php' );
		} else {
			// Router
			switch ( $this->active_tab ) {
				case 'cache':
					$this->safe_include( 'inc/admin/views/cache-settings.php' );
					break;
				case 'security':
					$this->safe_include( 'inc/admin/views/security-settings.php' );
					break;
				case 'translation':
					$this->safe_include( 'inc/admin/views/translation-settings.php' );
					break;
				case 'license':
					$this->safe_include( 'inc/admin/views/license-settings.php' );
					break;
				default:
					$this->safe_include( 'inc/admin/views/dashboard.php' );
					break;
			}
		}
		
		echo '</div>'; // End .uwsct-content
		echo '</div>'; // End .uwsct-container

		// AI Helper Bubble (Always loaded)
		$this->safe_include( 'inc/admin/views/ai-helper.php' );
	}

	/**
	 * Helper to safely include views and pass variables.
	 */
	private function safe_include( $path, $args = [] ) {
		// Extract variables to be used in the view
		if ( ! empty( $args ) ) {
			extract( $args );
		}
		
		$full_path = UWSCT_PLUGIN_DIR . $path;
		if ( file_exists( $full_path ) ) {
			include $full_path;
		} else {
			echo '<div class="notice notice-error"><p>Error: Missing view file: ' . esc_html( $path ) . '</p></div>';
		}
	}
}