<?php
namespace UWSCT\Inc\Cache;

class Cache_Manager {

	public function __construct() {
		add_action( 'init', [ $this, 'init_modules' ] );
		add_action( 'admin_post_uwsct_save_cache_settings', [ $this, 'save_settings' ] );
	}

	public function init_modules() {
		if ( get_option( 'uwsct_enable_page_cache' ) ) {
			if ( file_exists( UWSCT_PLUGIN_DIR . 'inc/cache/class-page-caching.php' ) ) {
				require_once UWSCT_PLUGIN_DIR . 'inc/cache/class-page-caching.php';
				new Page_Caching();
			}
		}
		if ( get_option( 'uwsct_enable_minification' ) ) {
			if ( file_exists( UWSCT_PLUGIN_DIR . 'inc/cache/class-minification.php' ) ) {
				require_once UWSCT_PLUGIN_DIR . 'inc/cache/class-minification.php';
				new Minification();
			}
		}
        
        // Cloudflare Integration
        if ( get_option( 'uwsct_enable_cloudflare' ) ) {
            if ( file_exists( UWSCT_PLUGIN_DIR . 'inc/cache/class-cdn-integrator.php' ) ) {
				require_once UWSCT_PLUGIN_DIR . 'inc/cache/class-cdn-integrator.php';
				new CDN_Integrator();
			}
        }

		// Bloat Removal
		if ( get_option( 'uwsct_disable_emojis' ) ) {
			remove_action( 'wp_head', 'print_emoji_detection_script', 7 );
			remove_action( 'wp_print_styles', 'print_emoji_styles' );
		}
		if ( get_option( 'uwsct_disable_embeds' ) ) {
			wp_deregister_script( 'wp-embed' );
		}
	}

	public function save_settings() {
		if ( ! current_user_can( 'manage_options' ) ) wp_die( 'Unauthorized access.' );
		check_admin_referer( 'uwsct_cache_settings_nonce' );

		// Core toggles
		update_option( 'uwsct_enable_page_cache', isset( $_POST['uwsct_enable_page_cache'] ) ? 1 : 0 );
		update_option( 'uwsct_enable_minification', isset( $_POST['uwsct_enable_minification'] ) ? 1 : 0 );
		update_option( 'uwsct_enable_gzip', isset( $_POST['uwsct_enable_gzip'] ) ? 1 : 0 );
		update_option( 'uwsct_enable_browser_cache', isset( $_POST['uwsct_enable_browser_cache'] ) ? 1 : 0 );
		
		// Advanced Toggles
		update_option( 'uwsct_cache_mobile', isset( $_POST['uwsct_cache_mobile'] ) ? 1 : 0 );
		update_option( 'uwsct_cache_loggedin', isset( $_POST['uwsct_cache_loggedin'] ) ? 1 : 0 );
        
        // Exclusions
        if ( isset( $_POST['uwsct_cache_excluded_ids'] ) && is_array( $_POST['uwsct_cache_excluded_ids'] ) ) {
            $ids = array_map( 'intval', $_POST['uwsct_cache_excluded_ids'] );
            update_option( 'uwsct_cache_excluded_ids', $ids );
        } else {
            update_option( 'uwsct_cache_excluded_ids', [] );
        }
        if ( isset( $_POST['uwsct_cache_excluded_paths'] ) ) {
            update_option( 'uwsct_cache_excluded_paths', sanitize_textarea_field( $_POST['uwsct_cache_excluded_paths'] ) );
        }
        
        // Cloudflare Settings
        update_option( 'uwsct_enable_cloudflare', isset( $_POST['uwsct_enable_cloudflare'] ) ? 1 : 0 );
        if ( isset( $_POST['uwsct_cf_api_key'] ) ) update_option( 'uwsct_cf_api_key', sanitize_text_field( $_POST['uwsct_cf_api_key'] ) );
        if ( isset( $_POST['uwsct_cf_email'] ) ) update_option( 'uwsct_cf_email', sanitize_email( $_POST['uwsct_cf_email'] ) );
        if ( isset( $_POST['uwsct_cf_zone_id'] ) ) update_option( 'uwsct_cf_zone_id', sanitize_text_field( $_POST['uwsct_cf_zone_id'] ) );

		// Value Inputs
		if ( isset( $_POST['uwsct_cache_lifespan'] ) ) {
            update_option( 'uwsct_cache_lifespan', intval( $_POST['uwsct_cache_lifespan'] ) );
        }

        // Trigger .htaccess updates safely
        ob_start();
        try {
            $this->update_htaccess_rules();
        } catch ( \Exception $e ) {
            error_log( 'UWSCT .htaccess error: ' . $e->getMessage() );
        }
        ob_end_clean();
        
        // Test Cloudflare Connection if Enabled
        $cf_status = '';
        if ( get_option( 'uwsct_enable_cloudflare' ) ) {
             if ( file_exists( UWSCT_PLUGIN_DIR . 'inc/cache/class-cdn-integrator.php' ) ) {
				require_once UWSCT_PLUGIN_DIR . 'inc/cache/class-cdn-integrator.php';
				$cdn = new CDN_Integrator();
                $is_valid = $cdn->test_connection();
                $cf_status = $is_valid ? '&cf_status=success' : '&cf_status=error';
			}
        }

		wp_redirect( admin_url( 'admin.php?page=uwsct_dashboard&tab=cache&status=saved' . $cf_status ) );
		exit;
	}

    private function update_htaccess_rules() {
        $htaccess_path = ABSPATH . '.htaccess';
        if ( ! file_exists( $htaccess_path ) || ! is_writable( $htaccess_path ) ) return;

        $rules = "\n# BEGIN UWSCT Optimization\n";

        if ( get_option( 'uwsct_enable_gzip' ) ) {
            $rules .= "<IfModule mod_deflate.c>\n";
            $rules .= "AddOutputFilterByType DEFLATE text/html text/plain text/xml text/css text/javascript application/javascript application/x-javascript application/json application/xml\n";
            $rules .= "</IfModule>\n";
        }

        if ( get_option( 'uwsct_enable_browser_cache' ) ) {
            $rules .= "<IfModule mod_expires.c>\n";
            $rules .= "ExpiresActive On\n";
            $rules .= "ExpiresByType image/jpg \"access plus 1 year\"\n";
            $rules .= "ExpiresByType image/jpeg \"access plus 1 year\"\n";
            $rules .= "ExpiresByType image/gif \"access plus 1 year\"\n";
            $rules .= "ExpiresByType image/png \"access plus 1 year\"\n";
            $rules .= "ExpiresByType text/css \"access plus 1 month\"\n";
            $rules .= "ExpiresByType application/pdf \"access plus 1 month\"\n";
            $rules .= "ExpiresByType text/x-javascript \"access plus 1 month\"\n";
            $rules .= "ExpiresByType application/x-shockwave-flash \"access plus 1 month\"\n";
            $rules .= "ExpiresByType image/x-icon \"access plus 1 year\"\n";
            $rules .= "ExpiresDefault \"access plus 2 days\"\n";
            $rules .= "</IfModule>\n";
        }

        $rules .= "# END UWSCT Optimization\n";

        $content = file_get_contents( $htaccess_path );
        if ( preg_match( '/# BEGIN UWSCT Optimization.*# END UWSCT Optimization\s*/s', $content ) ) {
            $content = preg_replace( '/# BEGIN UWSCT Optimization(.*?)# END UWSCT Optimization\s*/s', '', $content );
        }
        file_put_contents( $htaccess_path, $rules . $content );
    }
}