<?php
namespace UWSCT\Inc\Cache;

class Page_Caching {

	private $cache_path;
	private $default_exclusions = [];

	public function __construct() {
		$this->cache_path = WP_CONTENT_DIR . '/cache/uwsct/';
		
		// 1. Hardcoded System Exclusions
		$this->default_exclusions = [
			'/wp-admin',
			'/wp-login.php',
			'xmlrpc.php',
            '/wc-api', // WooCommerce API
		];

		if ( ! is_dir( $this->cache_path ) ) {
			wp_mkdir_p( $this->cache_path );
			file_put_contents( $this->cache_path . 'index.php', '<?php // Silence is golden' );
		}

		add_action( 'template_redirect', [ $this, 'handle_cache_serving' ], 0 );
		add_action( 'save_post', [ $this, 'clear_cache' ] );
		add_action( 'comment_post', [ $this, 'clear_cache' ] );
		add_action( 'woocommerce_reduce_order_stock', [ $this, 'clear_cache' ] );
	}

	public function handle_cache_serving() {
		if ( $this->should_skip_cache() ) {
			return;
		}

		if ( $this->serve_cache() ) {
			exit;
		}

		ob_start( [ $this, 'create_cache_file' ] );
	}

	private function should_skip_cache() {
		// 1. Standard Checks (Logged in, POST request)
		if ( is_user_logged_in() && ! get_option( 'uwsct_cache_loggedin' ) ) return true;
		if ( $_SERVER['REQUEST_METHOD'] !== 'GET' ) return true;

		// 2. System Exclusions (Admin, API)
		$request_uri = $_SERVER['REQUEST_URI'];
		foreach ( $this->default_exclusions as $excluded ) {
			if ( strpos( $request_uri, $excluded ) !== false ) return true;
		}
        
        // 3. Dynamic WooCommerce Exclusions (Cart, Checkout, Account)
        // We check these if WooCommerce is active
        if ( function_exists( 'is_woocommerce' ) ) {
            if ( is_cart() || is_checkout() || is_account_page() ) {
                return true;
            }
        }

        // 4. User-Selected Page ID Exclusions
        $excluded_ids = get_option( 'uwsct_cache_excluded_ids', [] );
        if ( ! empty( $excluded_ids ) ) {
            // Use get_queried_object_id() to handle Pages, Posts, and CPTs
            $current_id = get_queried_object_id();
            if ( $current_id && in_array( $current_id, $excluded_ids ) ) {
                return true;
            }
        }

        // 5. Custom URL Path Exclusions
        $custom_paths = get_option( 'uwsct_cache_excluded_paths', '' );
        if ( ! empty( $custom_paths ) ) {
            $paths = array_filter( array_map( 'trim', explode( "\n", $custom_paths ) ) );
            foreach ( $paths as $path ) {
                if ( empty( $path ) ) continue;
                if ( strpos( $request_uri, $path ) !== false ) {
                    return true;
                }
            }
        }

		// 6. Query String check (allow translation param)
		if ( ! empty( $_SERVER['QUERY_STRING'] ) && strpos( $_SERVER['QUERY_STRING'], 'lang=' ) === false ) {
			return true;
		}

		// 7. Mobile Check
		if ( wp_is_mobile() && ! get_option( 'uwsct_cache_mobile' ) ) {
			return true;
		}

		return false;
	}

	public function create_cache_file( $buffer ) {
		if ( empty( $buffer ) || is_404() || is_search() ) return $buffer;

		$time = date('Y-m-d H:i:s');
		$signature = "\n<!-- Optimized by Ultimate WP SCT on {$time} -->";
		$final_content = $buffer . $signature;

		$file_name = $this->get_cache_filename();
		file_put_contents( $file_name, $final_content, LOCK_EX );

		return $final_content;
	}

	public function serve_cache() {
		$file_name = $this->get_cache_filename();
		$lifespan = get_option( 'uwsct_cache_lifespan', 10 ) * HOUR_IN_SECONDS; 

		if ( file_exists( $file_name ) && ( time() - filemtime( $file_name ) < $lifespan ) ) {
			header( 'Cache-Control: public, max-age=86400' );
			header( 'Pragma: cache' );
			header( 'Expires: ' . gmdate( 'D, d M Y H:i:s', time() + 86400 ) . ' GMT' );
			header( 'Content-Type: text/html; charset=UTF-8' );
			header( 'X-UWSCT-Cache: HIT' );
			readfile( $file_name );
			return true;
		}
		return false;
	}

	private function get_cache_filename() {
		$mobile_suffix = ( wp_is_mobile() && get_option('uwsct_cache_mobile') ) ? '_mobile' : '';
		$url_key = $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
		$url_hash = md5( $url_key );
		return $this->cache_path . $url_hash . $mobile_suffix . '.html';
	}

	public function clear_cache() {
		$files = glob( $this->cache_path . '*.html' );
		if ( $files ) {
			foreach ( $files as $file ) {
				@unlink( $file );
			}
		}
	}
}