<?php
/**
 * AI Agent Manager (Self-Healing)
 *
 * Automatically detects available Gemini models associated with the API key
 * to prevent "Model Not Found" errors.
 *
 * @package    Ultimate_WP_SCT
 * @subpackage Ultimate_WP_SCT/inc/common
 */

namespace UWSCT\Inc\Common;

class AI_Agent {

    const API_BASE = 'https://generativelanguage.googleapis.com/v1beta';

	public function __construct() {
		add_action( 'wp_ajax_uwsct_ai_query', [ $this, 'handle_ai_query' ] );
		add_action( 'admin_post_uwsct_save_dashboard_settings', [ $this, 'save_settings' ] );
	}

    public function save_settings() {
        if ( ! current_user_can( 'manage_options' ) ) wp_die( 'Unauthorized' );
        check_admin_referer( 'uwsct_dashboard_settings_nonce' );

        if ( isset( $_POST['uwsct_gemini_api_key'] ) ) {
            update_option( 'uwsct_gemini_api_key', sanitize_text_field( $_POST['uwsct_gemini_api_key'] ) );
            // Clear cached model on key change so we re-discover for the new key
            delete_transient( 'uwsct_gemini_model_endpoint' );
        }

        wp_redirect( admin_url( 'admin.php?page=uwsct_dashboard&status=saved' ) );
        exit;
    }

	public function handle_ai_query() {
		check_ajax_referer( 'uwsct_admin_nonce', 'nonce' );
		if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error( [ 'message' => 'Unauthorized' ] );

		$user_query = sanitize_text_field( $_POST['query'] );
		$api_key = trim( get_option( 'uwsct_gemini_api_key' ) );

		if ( empty( $api_key ) ) {
			wp_send_json_error( [ 'message' => 'Please enter your Google Gemini API Key in Dashboard settings.' ] );
		}

        // 1. Get Valid Model Endpoint (Cached)
        $model_endpoint = get_transient( 'uwsct_gemini_model_endpoint' );
        
        // If no cached model, discover one
        if ( ! $model_endpoint ) {
            $model_endpoint = $this->discover_best_model( $api_key );
            if ( is_wp_error( $model_endpoint ) ) {
                wp_send_json_error( [ 'message' => $model_endpoint->get_error_message() ] );
            }
            // Cache the working URL for 12 hours
            set_transient( 'uwsct_gemini_model_endpoint', $model_endpoint, 12 * HOUR_IN_SECONDS );
        }

		// 2. Construct Prompt
		$system_instruction = "You are the support agent for the 'Ultimate WordPress Security, Cache and Translation' plugin.
		Features: Page Caching, Minification, GZIP, Browser Cache, WAF, Login Guard, File Scan, 2FA, Translation Widget.
		Licensing: 30-day free trial, then key required (Product ID 26220).
		Style: Concise, helpful, use HTML formatting (bold, br).
		USER QUESTION: " . $user_query;

		// 3. Call API
		$response = wp_remote_post( $model_endpoint . '?key=' . $api_key, [
			'headers' => [ 'Content-Type' => 'application/json' ],
			'body'    => json_encode([
				'contents' => [[ 'parts' => [[ 'text' => $system_instruction ]] ]]
			]),
			'timeout' => 25
		]);

		if ( is_wp_error( $response ) ) {
            // If request failed, clear cache to retry discovery next time
            delete_transient( 'uwsct_gemini_model_endpoint' );
			wp_send_json_error( [ 'message' => 'Connection Error: ' . $response->get_error_message() ] );
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( isset( $body['error'] ) ) {
            // If API returns error, clear cache (maybe model was deprecated)
            delete_transient( 'uwsct_gemini_model_endpoint' ); 
            wp_send_json_error( [ 'message' => 'Gemini Error: ' . $body['error']['message'] ] );
        }

		if ( isset( $body['candidates'][0]['content']['parts'][0]['text'] ) ) {
			$answer = wp_kses_post( $body['candidates'][0]['content']['parts'][0]['text'] ); 
			wp_send_json_success( [ 'answer' => $answer ] );
		} else {
			wp_send_json_error( [ 'message' => 'AI returned no text. Check safety settings or key.' ] );
		}
	}

    /**
     * Auto-detects which model the API Key has access to.
     */
    private function discover_best_model( $api_key ) {
        // Ask Google: "What models can I use?"
        $url = self::API_BASE . '/models?key=' . $api_key;
        $response = wp_remote_get( $url );

        if ( is_wp_error( $response ) ) {
            return new \WP_Error( 'connect_fail', 'Could not connect to Google to list models.' );
        }

        $data = json_decode( wp_remote_retrieve_body( $response ), true );
        
        if ( isset( $data['error'] ) ) {
            return new \WP_Error( 'api_error', 'API Key Error: ' . $data['error']['message'] );
        }

        if ( empty( $data['models'] ) ) {
            return new \WP_Error( 'no_models', 'No models found for this API key.' );
        }

        // Preference list: Newest/Fastest -> Legacy Stable
        $preferences = [ 
            'gemini-1.5-flash', 
            'gemini-1.5-pro', 
            'gemini-1.0-pro', 
            'gemini-pro' 
        ];
        
        $found_model = '';

        // 1. Try to find a preferred model from our list
        foreach ( $preferences as $pref ) {
            foreach ( $data['models'] as $model ) {
                // Match model name (e.g. "models/gemini-1.5-flash-001")
                if ( strpos( $model['name'], $pref ) !== false && in_array( 'generateContent', $model['supportedGenerationMethods'] ) ) {
                    $found_model = $model['name'];
                    break 2;
                }
            }
        }

        // 2. If no preferred model, take ANY model that supports generateContent
        if ( empty( $found_model ) ) {
            foreach ( $data['models'] as $model ) {
                if ( in_array( 'generateContent', $model['supportedGenerationMethods'] ) ) {
                    $found_model = $model['name'];
                    break;
                }
            }
        }

        if ( ! empty( $found_model ) ) {
            // Return full URL: https://.../v1beta/models/MODEL_NAME:generateContent
            return self::API_BASE . '/' . $found_model . ':generateContent';
        }

        return new \WP_Error( 'no_chat_model', 'No chat-capable models found for this key.' );
    }
}