<?php
/**
 * Licensing and Trial Management Class
 *
 * Handles the 30-day free trial, license key verification, and
 * remote site fingerprinting to prevent trial abuse.
 *
 * @package    Ultimate_WP_SCT
 * @subpackage Ultimate_WP_SCT/inc/common
 */

namespace UWSCT\Inc\Common;

class Licensing {

	/**
	 * The API endpoint for the licensing server.
	 * REPLACE THIS with your actual licensing server REST API endpoint.
	 */
	const API_URL = 'https://www.ultimatemultimediaconsult.com/wp-json/uwsct-api/v1';

	/**
	 * Product ID as defined in the prompt.
	 * This MUST match the ID expected by your server/license manager.
	 */
	const PRODUCT_ID = '26220';

	/**
	 * Trial period in seconds (30 days).
	 */
	const TRIAL_PERIOD = 30 * DAY_IN_SECONDS;

	/**
	 * Option names.
	 */
	const OPTION_INSTALL_DATE  = 'uwsct_install_date';
	const OPTION_FINGERPRINT   = 'uwsct_site_fingerprint';
	const OPTION_LICENSE_KEY   = 'uwsct_license_key';
	const OPTION_LICENSE_STATUS = 'uwsct_license_status'; // 'valid', 'invalid', 'expired'

	/**
	 * Initialize the class.
	 */
	public function __construct() {
		add_action( 'admin_init', [ $this, 'register_settings' ] );
		add_action( 'admin_notices', [ $this, 'display_trial_notices' ] );
		
		// Handle License Activation AJAX
		add_action( 'wp_ajax_uwsct_activate_license', [ $this, 'ajax_activate_license' ] );
	}

	/**
	 * Check if the plugin functionality should be active.
	 * Returns true if:
	 * 1. License is 'valid' OR
	 * 2. Trial is still active.
	 *
	 * @return bool
	 */
	public function is_license_active_or_trial() {
		if ( $this->is_license_valid() ) {
			return true;
		}
		return $this->is_trial_active();
	}

	/**
	 * Check if a paid license is valid.
	 *
	 * @return bool
	 */
	public function is_license_valid() {
		$status = get_option( self::OPTION_LICENSE_STATUS );
		return 'valid' === $status;
	}

	/**
	 * Check if the 30-day trial is active.
	 *
	 * @return bool
	 */
	public function is_trial_active() {
		$install_date = get_option( self::OPTION_INSTALL_DATE );
		
		// If no install date exists (tampered db), we fail closed or try to recover via remote check
		if ( ! $install_date ) {
			return false; 
		}

		$time_diff = time() - (int) $install_date;
		return $time_diff < self::TRIAL_PERIOD;
	}

	/**
	 * Calculate remaining trial days.
	 *
	 * @return int Days remaining.
	 */
	public function get_trial_days_remaining() {
		$install_date = get_option( self::OPTION_INSTALL_DATE );
		if ( ! $install_date ) {
			return 0;
		}
		$seconds_passed = time() - (int) $install_date;
		$seconds_left   = self::TRIAL_PERIOD - $seconds_passed;
		
		return $seconds_left > 0 ? ceil( $seconds_left / DAY_IN_SECONDS ) : 0;
	}

	/**
	 * Register the License Key setting field.
	 */
	public function register_settings() {
		register_setting( 'uwsct_license_group', self::OPTION_LICENSE_KEY );
	}

	/**
	 * Display Admin Notices for Trial and Expiration.
	 */
	public function display_trial_notices() {
		// Don't show on public facing pages
		if ( ! is_admin() ) {
			return;
		}

		// If licensed, do nothing
		if ( $this->is_license_valid() ) {
			return;
		}

		$days_left = $this->get_trial_days_remaining();
		$upgrade_url = 'https://www.ultimatemultimediaconsult.com/product/ultimate-wordpress-security-cache-and-translation-plugin/';

		if ( $days_left > 0 ) {
			// Active Trial Notice
			?>
			<div class="notice notice-info is-dismissible">
				<p>
					<strong>Ultimate WP Security, Cache & Translation:</strong> 
					You are on the <strong>Free Premium Trial</strong>. You have <strong><?php echo intval( $days_left ); ?> days</strong> remaining.
					<a href="<?php echo esc_url( $upgrade_url ); ?>" target="_blank" class="button button-primary" style="margin-left: 10px;">Buy License Key</a>
				</p>
			</div>
			<?php
		} else {
			// Expired Trial Notice (Persistent, Red)
			?>
			<div class="notice notice-error">
				<p>
					<strong>TRIAL EXPIRED:</strong> Your 1-month free Premium trial has ended. 
					Upgrade now to continue securing and optimizing your site.
				</p>
				<p>
					<a href="<?php echo esc_url( $upgrade_url ); ?>" target="_blank" class="button button-primary">Get License Key</a>
					<a href="<?php echo admin_url( 'admin.php?page=uwsct_dashboard&tab=license' ); ?>" class="button button-secondary">Enter License Key</a>
				</p>
			</div>
			<?php
		}
	}

	/**
	 * Generate the Site Fingerprint.
	 * Used to identify the site uniquely even if they reinstall the plugin locally.
	 *
	 * @return string Hash
	 */
	public static function generate_fingerprint() {
		$url = get_site_url();
		$salt = wp_salt(); // WP specific salt
		return hash( 'sha256', $url . $salt . 'uwsct_secret' );
	}

	/**
	 * Registers the site with the remote licensing server.
	 * Called by Activator.
	 */
	public function register_site_remotely() {
		$fingerprint = get_option( self::OPTION_FINGERPRINT );
		$install_date = get_option( self::OPTION_INSTALL_DATE );

		// Prepare API Body
		$body = [
			'action'      => 'register_trial',
			'site_url'    => get_site_url(),
			'fingerprint' => $fingerprint,
			'install_date'=> $install_date,
			'product_id'  => self::PRODUCT_ID
		];

		// Send Remote Request (Fire and forget, or log error)
		$response = wp_remote_post( self::API_URL . '/register', [
			'body'    => $body,
			'timeout' => 15,
		]);

		if ( is_wp_error( $response ) ) {
			error_log( 'UWSCT Licensing Error: ' . $response->get_error_message() );
		}
	}

	/**
	 * Daily Cron Check to verify license validity/trial status from server.
	 */
	public function check_remote_license_status() {
		$license_key = get_option( self::OPTION_LICENSE_KEY );
		$fingerprint = get_option( self::OPTION_FINGERPRINT );

		$body = [
			'action'      => 'check_status',
			'license_key' => $license_key,
			'fingerprint' => $fingerprint,
			'product_id'  => self::PRODUCT_ID, // IMPORTANT: Sending Product ID
			'site_url'    => get_site_url()
		];

		$response = wp_remote_post( self::API_URL . '/check', [
			'body' => $body
		]);

		if ( ! is_wp_error( $response ) && 200 === wp_remote_retrieve_response_code( $response ) ) {
			$data = json_decode( wp_remote_retrieve_body( $response ), true );
			
			if ( isset( $data['status'] ) ) {
				// Update local status based on server response
				update_option( self::OPTION_LICENSE_STATUS, sanitize_text_field( $data['status'] ) );
				
				// If server says trial started earlier than local DB says (user reset DB), update local DB
				if ( isset( $data['original_install_date'] ) ) {
					$server_date = intval( $data['original_install_date'] );
					$local_date = (int) get_option( self::OPTION_INSTALL_DATE );
					
					// If server date is older, trust server to prevent trial reset abuse
					if ( $server_date < $local_date ) {
						update_option( self::OPTION_INSTALL_DATE, $server_date );
					}
				}
			}
		}
	}

	/**
	 * AJAX Handler to activate a license key entered by user.
	 */
	public function ajax_activate_license() {
		check_ajax_referer( 'uwsct_admin_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( [ 'message' => 'Unauthorized' ] );
		}

		$license_key = sanitize_text_field( $_POST['license_key'] );

		if ( empty( $license_key ) ) {
			wp_send_json_error( [ 'message' => 'License key cannot be empty.' ] );
		}

		// Remote Activation Call
		$body = [
			'action'      => 'activate',
			'license_key' => $license_key,
			'site_url'    => get_site_url(),
			'fingerprint' => get_option( self::OPTION_FINGERPRINT ),
			'product_id'  => self::PRODUCT_ID
		];

		$response = wp_remote_post( self::API_URL . '/activate', [ 'body' => $body ] );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( [ 'message' => 'Connection failed. Please try again.' ] );
		}

		$response_body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( isset( $response_body['success'] ) && $response_body['success'] ) {
			update_option( self::OPTION_LICENSE_KEY, $license_key );
			update_option( self::OPTION_LICENSE_STATUS, 'valid' );
			wp_send_json_success( [ 'message' => 'License activated successfully! Thank you.' ] );
		} else {
			$error_msg = isset( $response_body['message'] ) ? $response_body['message'] : 'Invalid license key.';
			wp_send_json_error( [ 'message' => $error_msg ] );
		}
	}
}