<?php
namespace UWSCT\Inc\Security;

class Firewall {

	private $log_file;

	public function __construct() {
		$upload_dir = wp_upload_dir();
		$this->log_file = $upload_dir['basedir'] . '/uwsct-logs/firewall.log';
		
		// Create log folder securely
		if ( ! is_dir( dirname( $this->log_file ) ) ) {
			wp_mkdir_p( dirname( $this->log_file ) );
			file_put_contents( dirname( $this->log_file ) . '/.htaccess', 'deny from all' );
		}

		$this->run_waf();
	}

	private function run_waf() {
		// 1. Whitelist Admins (Prevents self-lockout)
		// We check cookie presence lightly to avoid expensive DB calls on every request
		if ( isset( $_COOKIE[LOGGED_IN_COOKIE] ) && is_user_logged_in() && current_user_can( 'manage_options' ) ) {
			return;
		}

		$request_uri  = $_SERVER['REQUEST_URI'];
		$query_string = $_SERVER['QUERY_STRING'];
		$user_agent   = $_SERVER['HTTP_USER_AGENT'] ?? '';

		// 2. Block SQL Injection (Advanced Patterns)
		if ( preg_match( '/(union(\s+all)?|select\s+.*from|insert\s+into|update\s+.*set|delete\s+from|drop\s+table|truncate\s+table)/i', $query_string ) ) {
			$this->block_request( 'SQL Injection Attempt' );
		}

		// 3. Block XSS & Script Injection
		if ( preg_match( '/(<script|javascript:|vbscript:|onload=|onerror=|base64_decode|eval\(|alert\()/i', $query_string ) ) {
			$this->block_request( 'XSS/Script Injection Attempt' );
		}

		// 4. Block Directory Traversal & System Files
		if ( preg_match( '/(\.\.\/|\.\.\\|etc\/passwd|boot\.ini|win\.ini)/', $request_uri ) ) {
			$this->block_request( 'Directory Traversal Attempt' );
		}

		// 5. Block Aggressive Bots (User Agent)
		$bad_bots = [ 'havij', 'sqlmap', 'nikto', 'acunetix', 'netsparker', 'wpscan', 'masscan' ];
		foreach ( $bad_bots as $bot ) {
			if ( stripos( $user_agent, $bot ) !== false ) {
				$this->block_request( 'Malicious Scanner: ' . esc_html($bot) );
			}
		}
	}

	private function block_request( $reason ) {
		$this->log_block( $reason );
		header( 'HTTP/1.1 403 Forbidden' );
		header( 'Content-Type: text/html' );
		die( "
			<div style='font-family:sans-serif; text-align:center; padding:50px;'>
				<h1 style='color:#d63638;'>403 Security Forbidden</h1>
				<p>Your request was blocked by the <strong>Ultimate WP SCT Firewall</strong>.</p>
				<p style='background:#eee; display:inline-block; padding:10px; border-radius:5px;'><strong>Reason:</strong> " . esc_html( $reason ) . "</p>
				<p>If you believe this is a mistake, please contact the site administrator.</p>
			</div>
		" );
	}

	private function log_block( $reason ) {
		$entry = sprintf(
			"[%s] IP: %s | Reason: %s | URL: %s | UA: %s\n",
			date('Y-m-d H:i:s'),
			$_SERVER['REMOTE_ADDR'],
			$reason,
			$_SERVER['REQUEST_URI'],
			$_SERVER['HTTP_USER_AGENT'] ?? 'Unknown'
		);
		// Append to log file securely with file locking
		file_put_contents( $this->log_file, $entry, FILE_APPEND | LOCK_EX );
	}
}