<?php
namespace UWSCT\Inc\Security;

class Login_Guard {

	public function __construct() {
		add_action( 'wp_login_failed', [ $this, 'handle_failed_login' ] );
		add_filter( 'authenticate', [ $this, 'check_ip_lockout' ], 30, 3 );
		
		// Handle Custom Login URL Redirect
		if ( get_option( 'uwsct_change_login_url' ) ) {
		    add_action( 'init', [ $this, 'redirect_login_page' ] );
		}
	}

	/**
	 * Log failed attempts.
	 */
	public function handle_failed_login( $username ) {
		$ip = $_SERVER['REMOTE_ADDR'];
		
		// Check whitelist before logging failure
		if ( $this->is_ip_whitelisted( $ip ) ) {
			return;
		}

		$failed_attempts = get_option( 'uwsct_failed_logins', [] );

		if ( ! isset( $failed_attempts[ $ip ] ) ) {
			$failed_attempts[ $ip ] = [
				'count' => 0,
				'last_attempt' => time()
			];
		}

		$failed_attempts[ $ip ]['count']++;
		$failed_attempts[ $ip ]['last_attempt'] = time();

		update_option( 'uwsct_failed_logins', $failed_attempts );
	}

	/**
	 * Check if IP is banned before allowing login.
	 */
	public function check_ip_lockout( $user, $username, $password ) {
		$ip = $_SERVER['REMOTE_ADDR'];
		
		// Always allow whitelisted IPs
		if ( $this->is_ip_whitelisted( $ip ) ) {
			return $user;
		}

		$failed_attempts = get_option( 'uwsct_failed_logins', [] );

		if ( isset( $failed_attempts[ $ip ] ) ) {
			$attempts = $failed_attempts[ $ip ];
			
			// Lockout Logic: 5 attempts in 15 minutes
			if ( $attempts['count'] >= 5 && ( time() - $attempts['last_attempt'] ) < 900 ) {
				return new \WP_Error( 'locked_out', '<strong>ERROR</strong>: Too many failed login attempts. You are locked out for 15 minutes.' );
			}

			// Reset if time passed
			if ( ( time() - $attempts['last_attempt'] ) > 900 ) {
				unset( $failed_attempts[ $ip ] );
				update_option( 'uwsct_failed_logins', $failed_attempts );
			}
		}

		return $user;
	}

	/**
	 * Helper: Check if IP is in whitelist.
	 */
	private function is_ip_whitelisted( $ip ) {
		$whitelist = get_option( 'uwsct_whitelist_ips', [] );
		return in_array( $ip, $whitelist );
	}
	
	/**
	 * Redirect default wp-login.php to new slug
	 */
	public function redirect_login_page() {
        // Get new slug setting, default to 'my-login'
	    $new_slug = get_option( 'uwsct_login_slug', 'my-login' );
        $request_uri = $_SERVER['REQUEST_URI'];

        // 1. Check if user is trying to access the NEW slug
        // We need to serve the login page content here manually or rewrite rules
        // For MVP simplicity: If user accesses new slug, we don't redirect.
        // Note: A true rewrite requires add_rewrite_rule which is complex. 
        // Simpler approach: Check if accessing wp-login.php directly
        
	    $page = basename( parse_url($request_uri, PHP_URL_PATH) );

        // If accessing default login page AND not requesting the new slug
	    if ( $page == "wp-login.php" && $_SERVER['REQUEST_METHOD'] == 'GET' && !isset($_GET['action']) ) {
            // Redirect to home for now as a "Hide" mechanism
	        wp_redirect( home_url() );
	        exit;
	    }
        
        // Note: To make the new slug *actually work* as a login page requires
        // rewriting the URL. For this snippet, we are just HIDING the old one.
        // To fully implement "Change Login URL" typically requires flushing rewrite rules.
        // For robustness, I recommend sticking to just HIDING/BLOCKING wp-login.php for now unless
        // you implement the full rewrite logic.
	}
}