<?php
/**
 * Security Manager Logic
 *
 * Orchestrates the Firewall, Login Guard, Scanner, and 2FA modules.
 *
 * @package    Ultimate_WP_SCT
 * @subpackage Ultimate_WP_SCT/inc/security
 */

namespace UWSCT\Inc\Security;

class Security_Manager {

	public function __construct() {
		add_action( 'init', [ $this, 'init_modules' ] );
		add_action( 'admin_post_uwsct_save_security_settings', [ $this, 'save_settings' ] );
	}

	public function init_modules() {
		// Load Firewall
		if ( get_option( 'uwsct_enable_firewall' ) ) {
			if ( file_exists( UWSCT_PLUGIN_DIR . 'inc/security/class-firewall.php' ) ) {
				require_once UWSCT_PLUGIN_DIR . 'inc/security/class-firewall.php';
				new Firewall();
			}
		}

		// Load Login Protection (includes Custom Login URL logic)
		if ( file_exists( UWSCT_PLUGIN_DIR . 'inc/security/class-login-guard.php' ) ) {
			require_once UWSCT_PLUGIN_DIR . 'inc/security/class-login-guard.php';
			new Login_Guard();
		}
        
        // File Change Detection
        if ( get_option( 'uwsct_enable_file_change_detection' ) ) {
             if ( file_exists( UWSCT_PLUGIN_DIR . 'inc/security/class-scanner.php' ) ) {
                require_once UWSCT_PLUGIN_DIR . 'inc/security/class-scanner.php';
                new Scanner();
            }
        }
        
        // Disable XML-RPC
        if ( get_option( 'uwsct_disable_xmlrpc' ) ) {
            add_filter( 'xmlrpc_enabled', '__return_false' );
        }
        
        // 2FA Logic
         if ( get_option( 'uwsct_enable_2fa' ) ) {
            if ( file_exists( UWSCT_PLUGIN_DIR . 'inc/security/class-two-factor.php' ) ) {
                require_once UWSCT_PLUGIN_DIR . 'inc/security/class-two-factor.php';
                new Two_Factor();
            }
        }
	}

	public function save_settings() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( 'Unauthorized access.' );
		}
		check_admin_referer( 'uwsct_security_settings_nonce' );

		// Save Toggles
		update_option( 'uwsct_enable_firewall', isset( $_POST['uwsct_enable_firewall'] ) ? 1 : 0 );
		update_option( 'uwsct_enable_login_guard', isset( $_POST['uwsct_enable_login_guard'] ) ? 1 : 0 );
        update_option( 'uwsct_enable_file_change_detection', isset( $_POST['uwsct_enable_file_change_detection'] ) ? 1 : 0 );
        update_option( 'uwsct_disable_xmlrpc', isset( $_POST['uwsct_disable_xmlrpc'] ) ? 1 : 0 );
        update_option( 'uwsct_enable_2fa', isset( $_POST['uwsct_enable_2fa'] ) ? 1 : 0 );
        
        // Login URL Logic
        update_option( 'uwsct_change_login_url', isset( $_POST['uwsct_change_login_url'] ) ? 1 : 0 );
        if ( isset( $_POST['uwsct_login_slug'] ) ) {
            $slug = sanitize_title( $_POST['uwsct_login_slug'] );
            if ( empty( $slug ) ) $slug = 'my-login';
            update_option( 'uwsct_login_slug', $slug );
        }
		
		// Save Whitelists (Sanitize text area input)
		if ( isset( $_POST['uwsct_whitelist_ips'] ) ) {
			$ips = array_map( 'trim', explode( "\n", sanitize_textarea_field( $_POST['uwsct_whitelist_ips'] ) ) );
			update_option( 'uwsct_whitelist_ips', array_filter( $ips ) );
		}

		// Safe Redirect
		wp_redirect( admin_url( 'admin.php?page=uwsct_dashboard&tab=security&status=saved' ) );
		exit;
	}
}